# Complete DOCPLEX implementation - Retry Attempt 4

from docplex.mp.model import Model

def allergy_accommodation_optimization():
    # 1. MODEL & DATA SETUP
    mdl = Model(name="allergy_accommodation_optimization")
    
    # Data from the problem
    allergy_types = ['peanut', 'dairy', 'gluten']
    cities = ['NYC', 'LA', 'CHI']
    
    # Cost data from the database
    costs = {
        ('peanut', 'NYC'): 120.0,
        ('dairy', 'LA'): 100.0,
        ('gluten', 'CHI'): 90.0
    }
    
    # Budget and minimum resource requirement
    total_budget = 500000
    min_resource = 5
    
    # 2. VARIABLES
    # Decision variables: resources allocated for each allergy type in each city
    x = {(i, j): mdl.continuous_var(name=f"x_{i}_{j}", lb=0) for i in allergy_types for j in cities}
    
    # 3. OBJECTIVE FUNCTION
    # Minimize the total cost of providing allergy accommodations
    objective = mdl.sum(costs.get((i, j), 0) * x[i, j] for i in allergy_types for j in cities)
    mdl.minimize(objective)
    
    # 4. CONSTRAINTS
    
    # Budget Constraint: Total cost must not exceed the budget
    budget_constraint = mdl.sum(costs.get((i, j), 0) * x[i, j] for i in allergy_types for j in cities) <= total_budget
    mdl.add_constraint(budget_constraint, ctname="budget_constraint")
    
    # Minimum Resource Requirement: Each allergy type in each city must receive at least min_resource units
    for i in allergy_types:
        for j in cities:
            if (i, j) in costs:
                min_resource_constraint = x[i, j] >= min_resource
                mdl.add_constraint(min_resource_constraint, ctname=f"min_resource_{i}_{j}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in allergy_types:
            for j in cities:
                if (i, j) in costs:
                    value = solution.get_value(x[i, j])
                    print(f"x[{i}, {j}] = {value:.3f}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
    
    return mdl

# Execute the optimization
allergy_accommodation_optimization()