#!/usr/bin/env python3
"""
Gurobipy 12.0.2 Implementation for Coffee Shop Staff Allocation Optimization
"""

import gurobipy as gp
from gurobipy import GRB

def coffee_shop_optimization():
    """Optimize staff allocation for coffee shops during happy hours."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("coffee_shop_staff_allocation")
    
    # Data from the database
    total_amounts = [25.5, 30.0, 15.75]  # happy_hour_member.Total_amount
    membership_levels = [1, 2, 3]        # member.Level_of_membership
    max_staff_per_shop = [10, 12, 15]    # shop_staff_limits.Max_Staff_Per_Shop
    min_staff_per_shop = [2, 3, 4]       # shop_staff_limits.Min_Staff_Per_Shop
    
    # Number of shops
    n_shops = len(total_amounts)
    
    # CRITICAL: Validate array lengths before loops
    assert len(total_amounts) == len(membership_levels) == len(max_staff_per_shop) == len(min_staff_per_shop) == n_shops, "Array length mismatch"
    
    # 2. VARIABLES
    # Decision variable: Number of staff assigned to each shop
    x = {i: model.addVar(vtype=GRB.CONTINUOUS, name=f"x_{i}", lb=min_staff_per_shop[i], ub=max_staff_per_shop[i]) 
         for i in range(n_shops)}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the total weighted spending by members during happy hours
    model.setObjective(
        gp.quicksum(membership_levels[i] * total_amounts[i] * x[i] for i in range(n_shops)), 
        GRB.MAXIMIZE
    )
    
    # 4. CONSTRAINTS
    
    # Total Staff Budget Constraint
    model.addConstr(
        gp.quicksum(x[i] for i in range(n_shops)) <= 150, 
        name="total_staff_budget"
    )
    
    # Maximum Spending Limit per Member
    for i in range(n_shops):
        model.addConstr(
            total_amounts[i] <= 50, 
            name=f"max_spending_{i}"
        )
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in range(n_shops):
            print(f"Staff assigned to shop {i}: {x[i].x:.3f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
if __name__ == "__main__":
    coffee_shop_optimization()