# Complete PYOMO implementation - Retry Attempt 1

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def optimize_employee_assignment():
    # 1. MODEL CREATION
    model = pyo.ConcreteModel()
    
    # 2. DATA SETUP
    # Assuming the following data is available:
    employees = [1, 2, 3]
    companies = [1, 2, 3]
    
    # Years of experience for each employee
    experience = {1: 5, 2: 10, 3: 15}
    
    # Profit for each company
    profit = {1: 100000, 2: 200000, 3: 300000}
    
    # Maximum number of employees each company can hire
    max_employees = {1: 10, 2: 15, 3: 20}
    
    # Weights for the objective function
    weight_experience = 0.7
    weight_profit = 0.3
    
    # 3. SETS
    model.employees = pyo.Set(initialize=employees)
    model.companies = pyo.Set(initialize=companies)
    
    # 4. PARAMETERS
    model.experience = pyo.Param(model.employees, initialize=experience)
    model.profit = pyo.Param(model.companies, initialize=profit)
    model.max_employees = pyo.Param(model.companies, initialize=max_employees)
    
    # 5. VARIABLES
    model.x = pyo.Var(model.employees, model.companies, within=pyo.Binary)
    
    # 6. OBJECTIVE FUNCTION
    def obj_rule(model):
        return sum((weight_experience * model.experience[i] + weight_profit * model.profit[j]) * model.x[i, j] 
                   for i in model.employees for j in model.companies)
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.maximize)
    
    # 7. CONSTRAINTS
    # Employee Assignment Constraint: Each employee must be assigned to exactly one company
    def employee_assignment_rule(model, i):
        return sum(model.x[i, j] for j in model.companies) == 1
    model.employee_assignment_constraint = pyo.Constraint(model.employees, rule=employee_assignment_rule)
    
    # Company Capacity Constraint: The total number of employees assigned to each company cannot exceed its maximum capacity
    def company_capacity_rule(model, j):
        return sum(model.x[i, j] for i in model.employees) <= model.max_employees[j]
    model.company_capacity_constraint = pyo.Constraint(model.companies, rule=company_capacity_rule)
    
    # 8. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')
    
    # Optional: Set solver options
    solver.options['TimeLimit'] = 300  # 5 minutes
    solver.options['MIPGap'] = 0.01    # 1% gap
    
    # Solve the model
    results = solver.solve(model, tee=True)  # tee=True shows solver output
    
    # 9. RESULT PROCESSING
    # Check solver status
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print("Optimal solution found!")
        print(f"Optimal value: {pyo.value(model.objective)}")
        
        # Extract variable values
        print("\nVariable values:")
        for i in model.employees:
            for j in model.companies:
                if pyo.value(model.x[i, j]) > 0.5:  # Only print assignments
                    print(f"Employee {i} assigned to Company {j}")
        
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")
    
    return model

# Execute the optimization
if __name__ == "__main__":
    optimize_employee_assignment()