# Complete Optimization Problem and Solution: culture_company

## 1. Problem Context and Goals

### Context  
The culture company is focused on maximizing its annual profit by strategically allocating its resources between two key areas: book clubs and movie productions. The company has a total budget of $1,000,000 available for investments in these areas. Each investment category has specific operational parameters that must be adhered to. For book clubs, the company must invest at least $100,000 but no more than $500,000. Similarly, for movies, the minimum investment is $150,000, with a maximum cap of $600,000. Additionally, the company has set a minimum total profit target of $200,000 to ensure financial sustainability.

The expected profit per unit investment is $0.15 for book clubs and $0.20 for movies. These profit rates are based on historical data and industry standards, ensuring realistic and achievable targets. The company’s goal is to determine the optimal amount to invest in each category to maximize total profit while staying within the defined budget and investment limits.

### Goals  
The primary goal of this optimization problem is to maximize the company’s total annual profit by determining the best allocation of investments between book clubs and movies. Success is measured by achieving the highest possible profit, calculated as the sum of the expected profit from book clubs and movies, while ensuring all operational constraints are met. The company aims to make data-driven decisions that align with its financial goals and resource limitations.

## 2. Constraints    

The company must adhere to several constraints to ensure a balanced and feasible investment strategy:

1. **Budget Constraint**: The total investment in both book clubs and movies must not exceed the available budget of $1,000,000.
2. **Minimum Investment in Book Clubs**: The company must invest at least $100,000 in book clubs to maintain operations.
3. **Minimum Investment in Movies**: A minimum investment of $150,000 is required in movies to cover production costs.
4. **Maximum Investment in Book Clubs**: The investment in book clubs cannot exceed $500,000 to prevent over-allocation of resources.
5. **Maximum Investment in Movies**: The investment in movies is capped at $600,000 to ensure diversification of resources.
6. **Minimum Total Profit**: The combined profit from both book clubs and movies must be at least $200,000 to meet the company’s financial sustainability target.

These constraints ensure that the company’s investment strategy is both realistic and aligned with its operational and financial goals.

## 3. Available Data  

### Database Schema  
```sql
-- Iteration 1 Database Schema
-- Objective: Schema changes and configuration logic updates implemented to address missing optimization requirements and mapping gaps identified by the OR expert.

CREATE TABLE investment_profits (
  profit_book_club FLOAT,
  profit_movie FLOAT,
  x_book_club FLOAT,
  x_movie FLOAT
);
```

### Data Dictionary  
The `investment_profits` table contains the following columns, each with a specific business purpose and optimization role:

- **profit_book_club**: Represents the expected profit per unit investment in book clubs. This value is used as the objective coefficient for book club investments in the optimization problem.
- **profit_movie**: Represents the expected profit per unit investment in movies. This value is used as the objective coefficient for movie investments in the optimization problem.
- **x_book_club**: Represents the decision variable for the amount to invest in book clubs. This is a continuous variable that can take any value within the defined investment limits.
- **x_movie**: Represents the decision variable for the amount to invest in movies. This is a continuous variable that can take any value within the defined investment limits.

### Current Stored Values  
```sql
-- Iteration 1 Realistic Data
-- Generated by triple expert (business + data + optimization)
-- Values were determined based on industry standards, historical data, and realistic business scenarios to ensure the optimization problem is meaningful and solvable.

-- Realistic data for investment_profits
INSERT INTO investment_profits (profit_book_club, profit_movie, x_book_club, x_movie) VALUES (0.15, 0.2, 200000.0, 300000.0);
INSERT INTO investment_profits (profit_book_club, profit_movie, x_book_club, x_movie) VALUES (0.18, 0.22, 250000.0, 350000.0);
INSERT INTO investment_profits (profit_book_club, profit_movie, x_book_club, x_movie) VALUES (0.12, 0.18, 150000.0, 250000.0);
```

## 4. Mathematical Optimization Formulation

#### Decision Variables
- \( x_b \): Amount to invest in book clubs (continuous variable, in dollars)  
- \( x_m \): Amount to invest in movies (continuous variable, in dollars)  

#### Objective Function
Maximize the total annual profit:  
\[
\text{Maximize } Z = 0.15 \cdot x_b + 0.20 \cdot x_m
\]  
**Data Source Verification**:  
- \( 0.15 \) from `investment_profits.profit_book_club`  
- \( 0.20 \) from `investment_profits.profit_movie`  

#### Constraints
1. **Budget Constraint**: Total investment must not exceed the available budget.  
\[
x_b + x_m \leq 1,000,000
\]  
2. **Minimum Investment in Book Clubs**:  
\[
x_b \geq 100,000
\]  
3. **Minimum Investment in Movies**:  
\[
x_m \geq 150,000
\]  
4. **Maximum Investment in Book Clubs**:  
\[
x_b \leq 500,000
\]  
5. **Maximum Investment in Movies**:  
\[
x_m \leq 600,000
\]  
6. **Minimum Total Profit**: Combined profit must meet the target.  
\[
0.15 \cdot x_b + 0.20 \cdot x_m \geq 200,000
\]  
**Data Source Verification**:  
- \( 0.15 \) from `investment_profits.profit_book_club`  
- \( 0.20 \) from `investment_profits.profit_movie`  

This is a complete, immediately solvable LINEAR programming model with all numerical coefficients derived from the provided data.

## 5. Gurobipy Implementation

```python
# Complete GUROBIPY implementation - Retry Attempt 4

import gurobipy as gp
from gurobipy import GRB

def optimize_investment():
    # 1. MODEL & DATA SETUP
    model = gp.Model("culture_company_investment")
    
    # 2. VARIABLES
    x_b = model.addVar(vtype=GRB.CONTINUOUS, name="x_book_club", lb=100000, ub=500000)
    x_m = model.addVar(vtype=GRB.CONTINUOUS, name="x_movie", lb=150000, ub=600000)
    
    # 3. OBJECTIVE FUNCTION
    model.setObjective(0.15 * x_b + 0.20 * x_m, GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS
    # Budget Constraint
    model.addConstr(x_b + x_m <= 1000000, name="budget_constraint")
    
    # Minimum Total Profit Constraint
    model.addConstr(0.15 * x_b + 0.20 * x_m >= 200000, name="min_profit_constraint")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        print(f"Investment in Book Clubs: {x_b.x}")
        print(f"Investment in Movies: {x_m.x}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
        # Analyze the infeasibility
        model.computeIIS()
        model.write("model.ilp")
        print("IIS written to file 'model.ilp'")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Execute the optimization
optimize_investment()
```

### Execution Results
**Status**: INFEASIBLE
**Error**: Problem is infeasible
**Analysis**: Gurobipy identified the problem as infeasible, indicating that no solution satisfies all constraints. This is consistent with the results from other solvers.

## 6. DOCplex Implementation

```python
# Complete DOCPLEX implementation - Retry Attempt 4

from docplex.mp.model import Model

def optimize_investment():
    # 1. MODEL & DATA SETUP
    mdl = Model(name="culture_company_investment")
    
    # Data from the problem statement
    budget = 1000000
    min_book_club = 100000
    min_movie = 150000
    max_book_club = 500000
    max_movie = 600000
    min_total_profit = 200000
    profit_book_club = 0.15
    profit_movie = 0.20
    
    # 2. VARIABLES
    x_b = mdl.continuous_var(name="x_book_club", lb=min_book_club, ub=max_book_club)
    x_m = mdl.continuous_var(name="x_movie", lb=min_movie, ub=max_movie)
    
    # 3. OBJECTIVE FUNCTION
    total_profit = profit_book_club * x_b + profit_movie * x_m
    mdl.maximize(total_profit)
    
    # 4. CONSTRAINTS
    # Budget Constraint
    mdl.add_constraint(x_b + x_m <= budget, ctname="budget_constraint")
    
    # Minimum Total Profit Constraint
    mdl.add_constraint(profit_book_club * x_b + profit_movie * x_m >= min_total_profit, ctname="min_total_profit_constraint")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        print(f"Investment in Book Clubs: {solution.get_value(x_b)}")
        print(f"Investment in Movies: {solution.get_value(x_m)}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
        if mdl.solve_details.status == "infeasible":
            print("The problem is infeasible. Please check the constraints.")
        elif mdl.solve_details.status == "unbounded":
            print("The problem is unbounded. Please check the objective function and constraints.")
    
    return mdl

# Execute the optimization
optimize_investment()
```

### Execution Results
**Status**: INFEASIBLE
**Error**: Problem is infeasible
**Analysis**: DOCplex also concluded that the problem is infeasible, reinforcing the findings from Gurobipy.

## 7. Pyomo Implementation

```python
# Complete PYOMO implementation - Retry Attempt 4

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def culture_company_optimization():
    # 1. MODEL CREATION
    model = pyo.ConcreteModel()
    
    # 2. DATA SETUP
    profit_book_club = 0.15
    profit_movie = 0.20
    budget = 1000000
    min_book_club = 100000
    min_movie = 150000
    max_book_club = 500000
    max_movie = 600000
    min_total_profit = 200000
    
    # 3. VARIABLES
    model.x_b = pyo.Var(within=pyo.NonNegativeReals, bounds=(min_book_club, max_book_club))
    model.x_m = pyo.Var(within=pyo.NonNegativeReals, bounds=(min_movie, max_movie))
    
    # 4. OBJECTIVE FUNCTION
    def obj_rule(model):
        return profit_book_club * model.x_b + profit_movie * model.x_m
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.maximize)
    
    # 5. CONSTRAINTS
    def budget_constraint_rule(model):
        return model.x_b + model.x_m <= budget
    model.budget_constraint = pyo.Constraint(rule=budget_constraint_rule)
    
    def min_profit_constraint_rule(model):
        return profit_book_club * model.x_b + profit_movie * model.x_m >= min_total_profit
    model.min_profit_constraint = pyo.Constraint(rule=min_profit_constraint_rule)
    
    # 6. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')
    
    # Optional: Set solver options
    solver.options['TimeLimit'] = 300  # 5 minutes
    solver.options['MIPGap'] = 0.01    # 1% gap
    
    # Solve the model
    results = solver.solve(model, tee=True)  # tee=True shows solver output
    
    # 7. RESULT PROCESSING
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print("Optimal solution found!")
        print(f"Optimal value: {pyo.value(model.objective)}")
        
        # Extract variable values
        print("\nVariable values:")
        print(f"x_b (Book Clubs Investment): {pyo.value(model.x_b)}")
        print(f"x_m (Movies Investment): {pyo.value(model.x_m)}")
        
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")
    
    return model

# Execute the optimization
culture_company_optimization()
```

### Execution Results
**Status**: INFEASIBLE
**Error**: Problem is infeasible
**Analysis**: Pyomo confirmed the infeasibility of the problem, aligning with the results from Gurobipy and DOCplex.

## 8. Cross-Solver Analysis and Final Recommendation

### Solver Results Comparison

| Solver | Status | Optimal Value | Execution Time | Decision Variables | Retry Attempt |
|--------|--------|---------------|----------------|-------------------|---------------|
| Gurobipy | INFEASIBLE | N/A | 0.18s | N/A | 4 |
| Docplex | INFEASIBLE | N/A | 1.03s | N/A | 4 |
| Pyomo | INFEASIBLE | N/A | 0.92s | N/A | 4 |

### Solver Consistency Analysis
**Result**: All solvers produced consistent results ✓
**Consistent Solvers**: gurobipy, docplex, pyomo
**Solver Retry Summary**: gurobipy: 4 attempts, docplex: 4 attempts, pyomo: 4 attempts

### Final Recommendation
**Confidence Level**: HIGH
**Preferred Solver(s)**: multiple
**Reasoning**: All three solvers consistently identified the problem as infeasible, providing high confidence in the result. The issue likely lies in the constraints rather than the solvers.

### Business Interpretation
**Overall Strategy**: The problem is infeasible, meaning that the current constraints do not allow for any investment strategy that meets all requirements. This suggests a need to revisit the constraints or the budget.
**Objective Value Meaning**: The objective value represents the total annual profit from investments in book clubs and movies. An infeasible solution means no combination of investments can achieve the target profit under the given constraints.
**Resource Allocation Summary**: No feasible allocation of resources exists under the current constraints. Consider revising the budget or relaxing some constraints.
**Implementation Recommendations**: Review and adjust the constraints, particularly the minimum and maximum investment limits, or increase the budget to find a feasible solution.