# Complete PYOMO implementation - Retry Attempt 4

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def customer_complaints_optimization():
    # 1. MODEL CREATION
    model = pyo.ConcreteModel()
    
    # 2. DATA SETUP
    # Staff data
    staff_ids = [1, 2, 3]
    hourly_wages = {1: 25.0, 2: 30.0, 3: 35.0}
    max_hours = {1: 40.0, 2: 45.0, 3: 50.0}
    
    # Complaint data
    complaint_ids = [1, 2, 3]
    min_time = {1: 1.0, 2: 1.5, 3: 2.0}
    max_time = {1: 3.0, 2: 4.0, 3: 5.0}
    
    # Validate array lengths
    assert len(staff_ids) == len(hourly_wages) == len(max_hours), "Staff data length mismatch"
    assert len(complaint_ids) == len(min_time) == len(max_time), "Complaint data length mismatch"
    
    # 3. SETS
    model.staff = pyo.Set(initialize=staff_ids)
    model.complaints = pyo.Set(initialize=complaint_ids)
    
    # 4. PARAMETERS
    model.hourly_wage = pyo.Param(model.staff, initialize=hourly_wages)
    model.max_hours = pyo.Param(model.staff, initialize=max_hours)
    model.min_time = pyo.Param(model.complaints, initialize=min_time)
    model.max_time = pyo.Param(model.complaints, initialize=max_time)
    
    # 5. VARIABLES
    model.time_spent = pyo.Var(model.staff, model.complaints, within=pyo.NonNegativeReals)
    
    # 6. OBJECTIVE FUNCTION
    def obj_rule(model):
        return sum(model.hourly_wage[i] * model.time_spent[i, j] for i in model.staff for j in model.complaints)
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.minimize)
    
    # 7. CONSTRAINTS
    # Complaint Assignment Constraint
    def complaint_assignment_rule(model, j):
        return sum(model.time_spent[i, j] for i in model.staff) == 1
    model.complaint_assignment = pyo.Constraint(model.complaints, rule=complaint_assignment_rule)
    
    # Time Bounds Constraint
    def time_bounds_rule(model, i, j):
        return (model.min_time[j], model.time_spent[i, j], model.max_time[j])
    model.time_bounds = pyo.Constraint(model.staff, model.complaints, rule=time_bounds_rule)
    
    # Staff Working Hours Constraint
    def staff_hours_rule(model, i):
        return sum(model.time_spent[i, j] for j in model.complaints) <= model.max_hours[i]
    model.staff_hours = pyo.Constraint(model.staff, rule=staff_hours_rule)
    
    # 8. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')
    
    # Optional: Set solver options
    solver.options['TimeLimit'] = 300  # 5 minutes
    solver.options['MIPGap'] = 0.01    # 1% gap
    
    # Solve the model
    results = solver.solve(model, tee=True)  # tee=True shows solver output
    
    # 9. RESULT PROCESSING
    # Check solver status
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print("Optimal solution found!")
        print(f"Optimal value: {pyo.value(model.objective)}")
        
        # Extract variable values
        print("\nTime spent by staff on complaints:")
        for i in model.staff:
            for j in model.complaints:
                time_val = pyo.value(model.time_spent[i, j])
                if time_val > 1e-6:  # Only print non-zero values
                    print(f"Staff {i} spent {time_val:.3f} hours on complaint {j}")
        
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")
    
    return model

# Execute the optimization
if __name__ == "__main__":
    customer_complaints_optimization()