# Complete GUROBIPY implementation - Retry Attempt 1

import gurobipy as gp
from gurobipy import GRB

def decoration_competition_optimization():
    # 1. MODEL & DATA SETUP
    model = gp.Model("decoration_competition")
    
    # Data from the database
    members = [1, 2, 3]
    rounds = [1, 2, 3]
    
    # Ranking points for each member in each round
    rank_in_round = {
        (1, 1): 15,
        (2, 2): 20,
        (3, 3): 25
    }
    
    # Member participation limit
    member_limit = {m: 2 for m in members}
    
    # Round participant requirement
    round_requirement = {r: 2 for r in rounds}
    
    # CRITICAL: Validate array lengths before loops
    assert len(members) == len(member_limit), "Member data length mismatch"
    assert len(rounds) == len(round_requirement), "Round data length mismatch"
    
    # 2. VARIABLES
    # Binary decision variables: x[m, r] = 1 if member m is assigned to round r, 0 otherwise
    x = model.addVars(members, rounds, vtype=GRB.BINARY, name="x")
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the total ranking points
    model.setObjective(gp.quicksum(rank_in_round[m, r] * x[m, r] for m in members for r in rounds if (m, r) in rank_in_round), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS
    
    # Member Participation Limit: Each member can participate in at most 2 rounds
    for m in members:
        model.addConstr(gp.quicksum(x[m, r] for r in rounds) <= member_limit[m], name=f"member_limit_{m}")
    
    # Round Participant Requirement: Each round must have exactly 2 participants
    for r in rounds:
        model.addConstr(gp.quicksum(x[m, r] for m in members) == round_requirement[r], name=f"round_requirement_{r}")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for m in members:
            for r in rounds:
                if x[m, r].x > 0.5:
                    print(f"Member {m} assigned to Round {r}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Execute the optimization
decoration_competition_optimization()