# Complete PYOMO implementation

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def employee_hire_optimization():
    """Optimize employee hiring and allocation strategy across multiple shops."""
    
    # 1. MODEL CREATION
    model = pyo.ConcreteModel()
    
    # 2. DATA SETUP
    # Define shop IDs and their respective data
    shop_ids = [1, 2, 3]
    
    # Cost per full-time and part-time employee for each shop
    cost_full_time = {1: 1500, 2: 1600, 3: 1700}
    cost_part_time = {1: 800, 2: 850, 3: 900}
    
    # Minimum staffing requirements for each shop
    min_staff = {1: 5, 2: 6, 3: 7}
    
    # Maximum allowed full-time and part-time employees for each shop
    max_full_time = {1: 10, 2: 12, 3: 15}
    max_part_time = {1: 8, 2: 10, 3: 12}
    
    # CRITICAL: Validate array lengths before indexing
    assert len(shop_ids) == len(cost_full_time) == len(cost_part_time) == len(min_staff) == len(max_full_time) == len(max_part_time), "Array length mismatch"
    
    # 3. SETS (Pyomo way to define indices)
    model.I = pyo.Set(initialize=shop_ids)  # Set of shop IDs
    
    # 4. PARAMETERS (data containers)
    model.cost_full_time = pyo.Param(model.I, initialize=cost_full_time)
    model.cost_part_time = pyo.Param(model.I, initialize=cost_part_time)
    model.min_staff = pyo.Param(model.I, initialize=min_staff)
    model.max_full_time = pyo.Param(model.I, initialize=max_full_time)
    model.max_part_time = pyo.Param(model.I, initialize=max_part_time)
    
    # 5. VARIABLES
    # Number of full-time and part-time employees to hire for each shop
    model.x_full_time = pyo.Var(model.I, within=pyo.NonNegativeIntegers)
    model.x_part_time = pyo.Var(model.I, within=pyo.NonNegativeIntegers)
    
    # 6. OBJECTIVE FUNCTION
    # Minimize the total hiring cost
    def obj_rule(model):
        return sum(model.cost_full_time[i] * model.x_full_time[i] + model.cost_part_time[i] * model.x_part_time[i] for i in model.I)
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.minimize)
    
    # 7. CONSTRAINTS
    
    # Minimum Staffing Requirement
    def min_staff_rule(model, i):
        return model.x_full_time[i] + model.x_part_time[i] >= model.min_staff[i]
    model.min_staff_constraint = pyo.Constraint(model.I, rule=min_staff_rule)
    
    # Maximum Full-Time Employees
    def max_full_time_rule(model, i):
        return model.x_full_time[i] <= model.max_full_time[i]
    model.max_full_time_constraint = pyo.Constraint(model.I, rule=max_full_time_rule)
    
    # Maximum Part-Time Employees
    def max_part_time_rule(model, i):
        return model.x_part_time[i] <= model.max_part_time[i]
    model.max_part_time_constraint = pyo.Constraint(model.I, rule=max_part_time_rule)
    
    # 8. SOLVING WITH GUROBI (your available solver)
    solver = SolverFactory('gurobi')
    
    # Optional: Set solver options
    solver.options['TimeLimit'] = 300  # 5 minutes
    solver.options['MIPGap'] = 0.01    # 1% gap
    
    # Solve the model
    results = solver.solve(model, tee=True)  # tee=True shows solver output
    
    # 9. RESULT PROCESSING
    # Check solver status
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print("Optimal solution found!")
        print(f"Optimal value: {pyo.value(model.objective)}")
        
        # Extract variable values
        print("\nVariable values:")
        for i in model.I:
            x_ft_val = pyo.value(model.x_full_time[i])
            x_pt_val = pyo.value(model.x_part_time[i])
            print(f"Shop {i}: Full-Time = {int(x_ft_val)}, Part-Time = {int(x_pt_val)}")
        
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")
    
    return model

# Run the optimization
if __name__ == "__main__":
    employee_hire_optimization()