#!/usr/bin/env python3
"""
DOCPLEX implementation for airline flight optimization problem
"""

from docplex.mp.model import Model

def optimize_flight_operations():
    """Optimize flight operations to minimize total cost"""
    
    # 1. MODEL & DATA SETUP
    mdl = Model(name="flight_optimization")
    
    # Sample data (replace with actual data loading)
    flight_costs = {101: 5000, 102: 6000, 103: 5500}  # Operational cost per flight
    employee_salaries = {1: 3000, 2: 3500, 3: 3200}  # Salary per employee
    
    # Flight capacity data
    aircraft_capacity = {
        ('JFK', 'LAX'): 8,
        ('LAX', 'JFK'): 8,
        ('ORD', 'DFW'): 5
    }
    
    # Employee assignment data
    employee_assignment = [
        (1, 101),
        (2, 102),
        (3, 103)
    ]
    
    # Extract unique flights and employees
    flights = list(flight_costs.keys())
    employees = list(employee_salaries.keys())
    
    # Validate array lengths
    assert len(flight_costs) == len(flights), "Flight cost data mismatch"
    assert len(employee_salaries) == len(employees), "Employee salary data mismatch"
    
    # 2. VARIABLES
    # Decision variables
    x = {f: mdl.binary_var(name=f"x_{f}") for f in flights}  # Flight operation
    y = {(e, f): mdl.binary_var(name=f"y_{e}_{f}") for e in employees for f in flights}  # Employee assignment
    
    # 3. OBJECTIVE FUNCTION
    # Minimize total cost: flight operational costs + employee salaries
    total_flight_cost = mdl.sum(flight_costs[f] * x[f] for f in flights)
    total_employee_cost = mdl.sum(employee_salaries[e] * y[(e, f)] for e in employees for f in flights)
    mdl.minimize(total_flight_cost + total_employee_cost)
    
    # 4. CONSTRAINTS
    
    # Flight Capacity Constraint
    # For each origin-destination pair, sum of flights <= max_flights
    # Assuming flights are grouped by origin-destination (simplified for example)
    for (origin, dest), max_flights in aircraft_capacity.items():
        relevant_flights = [f for f in flights if (origin, dest) in aircraft_capacity]  # Simplified
        mdl.add_constraint(mdl.sum(x[f] for f in relevant_flights) <= max_flights, ctname=f"capacity_{origin}_{dest}")
    
    # Employee Assignment Constraint
    # Each operated flight must have at least one assigned employee
    for f in flights:
        mdl.add_constraint(mdl.sum(y[(e, f)] for e in employees) >= x[f], ctname=f"staffing_{f}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for f in flights:
            if solution.get_value(x[f]) > 0.5:
                print(f"Flight {f} is operated")
                assigned_employees = [e for e in employees if solution.get_value(y[(e, f)]) > 0.5]
                print(f"Assigned employees: {assigned_employees}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
    
    return mdl

# Run the optimization
if __name__ == "__main__":
    optimize_flight_operations()