#!/usr/bin/env python3
"""
DOCplex implementation for university scholarship allocation optimization
"""

from docplex.mp.model import Model

def scholarship_allocation_optimization():
    """Optimize scholarship allocation based on academic performance, sports participation, and gaming habits."""
    
    # 1. MODEL & DATA SETUP
    mdl = Model(name="scholarship_allocation")
    
    # Data from the provided schema
    student_ids = [1, 2, 3]
    gpa = {1: 3.8, 2: 3.5, 3: 3.2}
    sports_hours = {1: 10.0, 2: 7.5, 3: 5.0}
    gaming_hours = {1: 12.5, 2: 15.0, 3: 10.0}
    min_sports_hours = {1: 2.0, 2: 2.5, 3: 3.0}
    max_sports_hours = {1: 15.0, 2: 18.0, 3: 20.0}
    min_gaming_hours = {1: 5.0, 2: 6.0, 3: 5.5}
    max_gaming_hours = {1: 20.0, 2: 25.0, 3: 22.0}
    weights = {'w1': 0.6, 'w2': 0.25, 'w3': 0.15}
    total_budget = 150000
    
    # CRITICAL: Validate array lengths to prevent IndexError
    assert len(student_ids) == len(gpa) == len(sports_hours) == len(gaming_hours) == len(min_sports_hours) == len(max_sports_hours) == len(min_gaming_hours) == len(max_gaming_hours), "Array length mismatch"
    safe_range = range(min(len(student_ids), len(gpa), len(sports_hours), len(gaming_hours), len(min_sports_hours), len(max_sports_hours), len(min_gaming_hours), len(max_gaming_hours)))  # Safe indexing
    
    # 2. VARIABLES
    scholarship_amount = {i: mdl.continuous_var(name=f"scholarship_{i}", lb=0) for i in safe_range}
    
    # 3. OBJECTIVE FUNCTION
    objective = mdl.sum((weights['w1'] * gpa[i] + weights['w2'] * sports_hours[i] + weights['w3'] * gaming_hours[i]) * scholarship_amount[i] for i in safe_range)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS
    
    # Total Budget Constraint
    total_scholarship = mdl.sum(scholarship_amount[i] for i in safe_range)
    mdl.add_constraint(total_scholarship <= total_budget, ctname="total_budget")
    
    # Scholarship Fairness Constraint
    for i in safe_range:
        mdl.add_constraint(scholarship_amount[i] >= 1000 * gpa[i], ctname=f"fairness_{i}")
    
    # Sports Participation Constraints
    for i in safe_range:
        mdl.add_constraint(sports_hours[i] >= min_sports_hours[i], ctname=f"min_sports_{i}")
        mdl.add_constraint(sports_hours[i] <= max_sports_hours[i], ctname=f"max_sports_{i}")
    
    # Gaming Habits Constraints
    for i in safe_range:
        mdl.add_constraint(gaming_hours[i] >= min_gaming_hours[i], ctname=f"min_gaming_{i}")
        mdl.add_constraint(gaming_hours[i] <= max_gaming_hours[i], ctname=f"max_gaming_{i}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in safe_range:
            value = solution.get_value(scholarship_amount[i])
            print(f"Scholarship for student {i}: {value:.2f}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
    
    return mdl

# Execute the optimization
if __name__ == "__main__":
    scholarship_allocation_optimization()