# Complete GUROBIPY implementation - Retry Attempt 2

import gurobipy as gp
from gurobipy import GRB

def optimize_reservations():
    # 1. MODEL & DATA SETUP
    model = gp.Model("HotelReservationOptimization")
    
    # Data from the Reservations table
    rates = [120.0, 100.0, 150.0, 110.0, 130.0]
    guests = [2, 3, 1, 2, 4]  # Assumed number of guests per reservation
    n_reservations = len(rates)
    
    # Validate array lengths before loops
    assert len(rates) == len(guests) == n_reservations, "Array length mismatch"
    
    # 2. VARIABLES
    x = {i: model.addVar(vtype=GRB.BINARY, name=f"x_{i}") for i in range(n_reservations)}
    
    # 3. OBJECTIVE FUNCTION
    model.setObjective(gp.quicksum(rates[i] * x[i] for i in range(n_reservations)), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS
    
    # Room Capacity Constraint
    model.addConstr(gp.quicksum(guests[i] * x[i] for i in range(n_reservations)) <= 4, name="room_capacity")
    
    # Date Overlap Constraint
    # Assuming the following overlapping pairs (i, j): (0, 1), (1, 2), (2, 3), (3, 4)
    overlapping_pairs = [(0, 1), (1, 2), (2, 3), (3, 4)]
    for i, j in overlapping_pairs:
        model.addConstr(x[i] + x[j] <= 1, name=f"no_overlap_{i}_{j}")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in range(n_reservations):
            if x[i].x > 1e-6:
                print(f"Reservation {i} accepted: {x[i].x:.0f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Execute the optimization
optimize_reservations()