# Complete PYOMO implementation

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def optimize_event_assignments():
    """Optimize participant assignments to events to maximize satisfaction."""
    
    # 1. MODEL CREATION
    model = pyo.ConcreteModel()
    
    # 2. DATA SETUP
    # Define sets for events and participants
    events = [1, 2, 3]
    participants = [101, 102, 103]
    
    # Satisfaction scores
    satisfaction_scores = {
        (1, 101): 0.85,
        (1, 102): 0.75,
        (1, 103): 0.65,
        (2, 101): 0.7,
        (2, 102): 0.9,
        (2, 103): 0.8,
        (3, 101): 0.6,
        (3, 102): 0.85,
        (3, 103): 0.95
    }
    
    # Event capacities
    event_capacities = {
        1: 50,
        2: 100,
        3: 75
    }
    
    # 3. SETS
    model.E = pyo.Set(initialize=events)  # Set of events
    model.P = pyo.Set(initialize=participants)  # Set of participants
    
    # 4. PARAMETERS
    model.satisfaction = pyo.Param(model.E, model.P, initialize=satisfaction_scores)
    model.capacity = pyo.Param(model.E, initialize=event_capacities)
    
    # 5. VARIABLES
    model.x = pyo.Var(model.E, model.P, within=pyo.Binary)  # Binary decision variables
    
    # 6. OBJECTIVE FUNCTION
    def obj_rule(model):
        return sum(model.satisfaction[e, p] * model.x[e, p] for e in model.E for p in model.P)
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.maximize)
    
    # 7. CONSTRAINTS
    
    # Event Capacity Constraint
    def event_capacity_rule(model, e):
        return sum(model.x[e, p] for p in model.P) <= model.capacity[e]
    model.event_capacity_constraint = pyo.Constraint(model.E, rule=event_capacity_rule)
    
    # Participant Assignment Constraint
    def participant_assignment_rule(model, p):
        return sum(model.x[e, p] for e in model.E) <= 1
    model.participant_assignment_constraint = pyo.Constraint(model.P, rule=participant_assignment_rule)
    
    # 8. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')
    
    # Optional: Set solver options
    solver.options['TimeLimit'] = 300  # 5 minutes
    solver.options['MIPGap'] = 0.01    # 1% gap
    
    # Solve the model
    results = solver.solve(model, tee=True)  # tee=True shows solver output
    
    # 9. RESULT PROCESSING
    # Check solver status
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print(f"Optimal value: {pyo.value(model.objective)}")
        
        # Extract variable values
        print("\nOptimal Assignments:")
        for e in model.E:
            for p in model.P:
                if pyo.value(model.x[e, p]) > 0.5:  # Only print assignments
                    print(f"Participant {p} assigned to Event {e}")
        
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")
    
    return model

# Execute the optimization
if __name__ == "__main__":
    optimize_event_assignments()