# Complete PYOMO implementation - Retry Attempt 4

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def machine_repair_optimization():
    # 1. MODEL CREATION
    model = pyo.ConcreteModel()
    
    # 2. DATA SETUP
    # Technicians and repairs
    technicians = [1, 2, 3]
    repairs = [101, 102, 103]
    
    # Repair time data
    repair_time_data = {
        (1, 101): 2.5,
        (2, 102): 3.0,
        (3, 103): 4.0
    }
    
    # Technician capacity data
    technician_capacity_data = {
        1: 5,
        2: 4,
        3: 3
    }
    
    # Machine priority data
    machine_priority_data = {
        201: 1,
        202: 2,
        203: 3
    }
    
    # Validate data before proceeding
    assert all((technician, repair) in repair_time_data for technician in technicians for repair in repairs), "Missing repair time data"
    assert all(technician in technician_capacity_data for technician in technicians), "Missing technician capacity data"
    assert all(repair in machine_priority_data for repair in repairs), "Missing machine priority data"
    
    # 3. SETS
    model.technicians = pyo.Set(initialize=technicians)
    model.repairs = pyo.Set(initialize=repairs)
    
    # 4. PARAMETERS
    model.repair_time = pyo.Param(model.technicians, model.repairs, initialize=repair_time_data)
    model.max_repairs = pyo.Param(model.technicians, initialize=technician_capacity_data)
    
    # 5. VARIABLES
    model.x = pyo.Var(model.technicians, model.repairs, within=pyo.Binary)
    
    # 6. OBJECTIVE FUNCTION
    def obj_rule(model):
        return sum(model.repair_time[i, j] * model.x[i, j] for i in model.technicians for j in model.repairs)
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.minimize)
    
    # 7. CONSTRAINTS
    # Single Assignment per Repair
    def single_assignment_rule(model, j):
        return sum(model.x[i, j] for i in model.technicians) == 1
    model.single_assignment = pyo.Constraint(model.repairs, rule=single_assignment_rule)
    
    # Technician Capacity Limit
    def capacity_rule(model, i):
        return sum(model.x[i, j] for j in model.repairs) <= model.max_repairs[i]
    model.capacity = pyo.Constraint(model.technicians, rule=capacity_rule)
    
    # 8. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')
    
    # Optional: Set solver options
    solver.options['TimeLimit'] = 300  # 5 minutes
    solver.options['MIPGap'] = 0.01    # 1% gap
    
    # Solve the model
    results = solver.solve(model, tee=True)  # tee=True shows solver output
    
    # 9. RESULT PROCESSING
    # Check solver status
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print("Optimal solution found!")
        print(f"Optimal value: {pyo.value(model.objective)}")
        
        # Extract variable values
        print("\nVariable values:")
        for i in model.technicians:
            for j in model.repairs:
                if pyo.value(model.x[i, j]) > 0:
                    print(f"Technician {i} assigned to Repair {j}")
        
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")
    
    return model

# Execute the optimization
if __name__ == "__main__":
    machine_repair_optimization()