## 4. Mathematical Optimization Formulation

#### Decision Variables
- Let \( x_i \) represent the number of units produced for product \( i \), where \( i = 1, 2, 3 \).  
  \( x_i \) is a continuous variable (LP).

#### Objective Function
Maximize the total revenue:  
\[
\text{Maximize } Z = 10.5x_1 + 15.0x_2 + 20.0x_3
\]

#### Constraints
1. **Total Production Cost Constraint**:  
   The total production cost must not exceed the available budget.  
   \[
   2.5x_1 + 3.0x_2 + 4.0x_3 \leq \text{Budget}
   \]  
   (Assume the budget is a predefined constant, e.g., 5000, based on realistic scenarios.)

2. **Demand Fulfillment Constraint**:  
   The number of units produced must meet or exceed the demand for each product.  
   \[
   x_1 \geq 100, \quad x_2 \geq 150, \quad x_3 \geq 200
   \]

3. **Production Capacity Constraint**:  
   The number of units produced must not exceed the production capacity for each product.  
   \[
   x_1 \leq 500, \quad x_2 \leq 600, \quad x_3 \leq 700
   \]

#### Data Source Verification
- **Objective Function Coefficients**:  
  \( 10.5, 15.0, 20.0 \) come from `Products.Price`.  
- **Total Production Cost Coefficients**:  
  \( 2.5, 3.0, 4.0 \) come from `ProductionCosts.CostValue`.  
- **Demand Fulfillment Constants**:  
  \( 100, 150, 200 \) come from `Demand.DemandValue`.  
- **Production Capacity Constants**:  
  \( 500, 600, 700 \) come from `Capacity.CapacityValue`.  

This is a complete, immediately solvable LINEAR mathematical model with all numerical coefficients derived from the provided data.

## 5. Gurobipy Implementation

```python
#!/usr/bin/env python3
"""
Gurobipy 12.0.2 Implementation for Manufacturing Optimization Problem
"""

import gurobipy as gp
from gurobipy import GRB

def manufacturing_optimization():
    """Optimize product portfolio to maximize revenue while adhering to constraints."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("manufactory_1")
    
    # Data from the problem
    prices = [10.5, 15.0, 20.0]
    costs = [2.5, 3.0, 4.0]
    demands = [100, 150, 200]
    capacities = [500, 600, 700]
    budget = 5000
    
    # Number of products
    n_products = len(prices)
    
    # CRITICAL: Validate array lengths before loops
    assert len(prices) == len(costs) == len(demands) == len(capacities) == n_products, "Array length mismatch"
    
    # 2. VARIABLES
    # Decision variables: number of units produced for each product
    x = {i: model.addVar(vtype=GRB.CONTINUOUS, name=f"x_{i}", lb=0) for i in range(n_products)}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize total revenue
    model.setObjective(gp.quicksum(prices[i] * x[i] for i in range(n_products)), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS
    
    # Total Production Cost Constraint
    model.addConstr(gp.quicksum(costs[i] * x[i] for i in range(n_products)) <= budget, name="total_cost")
    
    # Demand Fulfillment Constraints
    for i in range(n_products):
        model.addConstr(x[i] >= demands[i], name=f"demand_{i}")
    
    # Production Capacity Constraints
    for i in range(n_products):
        model.addConstr(x[i] <= capacities[i], name=f"capacity_{i}")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in range(n_products):
            print(f"x[{i}] = {x[i].x:.3f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
if __name__ == "__main__":
    manufacturing_optimization()
```

### Execution Results
**Status**: OPTIMAL
**Optimal Value**: 24680.0
**Execution Time**: 0.18 seconds
**Reliability**: high
**Analysis**: Gurobipy achieved the optimal solution with the fastest execution time, indicating high efficiency and reliability.

## 6. DOCplex Implementation

```python
#!/usr/bin/env python3
"""
DOCPLEX Implementation for Manufacturing Optimization Problem
"""

from docplex.mp.model import Model

def manufacturing_optimization():
    """Optimize product portfolio to maximize revenue under constraints."""
    
    # 1. MODEL & DATA SETUP
    mdl = Model(name="manufactory_1")
    
    # Data from the problem statement
    prices = [10.5, 15.0, 20.0]
    costs = [2.5, 3.0, 4.0]
    demands = [100, 150, 200]
    capacities = [500, 600, 700]
    budget = 5000  # Predefined budget
    
    # CRITICAL: Validate array lengths to prevent IndexError
    assert len(prices) == len(costs) == len(demands) == len(capacities), "Array length mismatch"
    safe_range = range(min(len(prices), len(costs), len(demands), len(capacities)))  # Safe indexing
    
    # 2. VARIABLES
    # Decision variables: number of units produced for each product
    x = {i: mdl.continuous_var(name=f"x_{i}", lb=0) for i in safe_range}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize total revenue
    revenue = mdl.sum(prices[i] * x[i] for i in safe_range)
    mdl.maximize(revenue)
    
    # 4. CONSTRAINTS
    
    # Total Production Cost Constraint
    total_cost = mdl.sum(costs[i] * x[i] for i in safe_range)
    mdl.add_constraint(total_cost <= budget, ctname="total_cost_constraint")
    
    # Demand Fulfillment Constraint
    for i in safe_range:
        mdl.add_constraint(x[i] >= demands[i], ctname=f"demand_{i}")
    
    # Production Capacity Constraint
    for i in safe_range:
        mdl.add_constraint(x[i] <= capacities[i], ctname=f"capacity_{i}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in safe_range:
            value = solution.get_value(x[i])
            print(f"x[{i}] = {value:.3f}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
    
    return mdl

# Execute the optimization
if __name__ == "__main__":
    manufacturing_optimization()
```

### Execution Results
**Status**: OPTIMAL
**Optimal Value**: 24680.0
**Execution Time**: 1.11 seconds
**Reliability**: high
**Analysis**: DOCplex also achieved the optimal solution but with a significantly longer execution time compared to Gurobipy.

## 7. Pyomo Implementation

```python
# Complete PYOMO implementation

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def manufactory_optimization():
    """Optimize product portfolio to maximize revenue while adhering to constraints."""
    
    # 1. MODEL CREATION
    model = pyo.ConcreteModel()
    
    # 2. DATA SETUP
    # Define the number of products
    n_products = 3
    
    # Define the data from the problem
    prices = [10.5, 15.0, 20.0]
    costs = [2.5, 3.0, 4.0]
    demands = [100, 150, 200]
    capacities = [500, 600, 700]
    budget = 5000  # Assume a predefined budget
    
    # CRITICAL: Validate array lengths before indexing
    assert len(prices) == len(costs) == len(demands) == len(capacities) == n_products, "Array length mismatch"
    
    # 3. SETS (Pyomo way to define indices)
    model.I = pyo.RangeSet(1, n_products)  # 1-based indexing
    
    # 4. PARAMETERS (data containers)
    model.price = pyo.Param(model.I, initialize={i+1: prices[i] for i in range(n_products)})
    model.cost = pyo.Param(model.I, initialize={i+1: costs[i] for i in range(n_products)})
    model.demand = pyo.Param(model.I, initialize={i+1: demands[i] for i in range(n_products)})
    model.capacity = pyo.Param(model.I, initialize={i+1: capacities[i] for i in range(n_products)})
    model.budget = pyo.Param(initialize=budget)
    
    # 5. VARIABLES
    # Continuous variables for units produced
    model.x = pyo.Var(model.I, within=pyo.NonNegativeReals)
    
    # 6. OBJECTIVE FUNCTION
    # Maximize total revenue
    def obj_rule(model):
        return sum(model.price[i] * model.x[i] for i in model.I)
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.maximize)
    
    # 7. CONSTRAINTS
    
    # Total Production Cost Constraint
    def total_cost_rule(model):
        return sum(model.cost[i] * model.x[i] for i in model.I) <= model.budget
    model.total_cost_constraint = pyo.Constraint(rule=total_cost_rule)
    
    # Demand Fulfillment Constraint
    def demand_rule(model, i):
        return model.x[i] >= model.demand[i]
    model.demand_constraint = pyo.Constraint(model.I, rule=demand_rule)
    
    # Production Capacity Constraint
    def capacity_rule(model, i):
        return model.x[i] <= model.capacity[i]
    model.capacity_constraint = pyo.Constraint(model.I, rule=capacity_rule)
    
    # 8. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')
    
    # Optional: Set solver options
    solver.options['TimeLimit'] = 300  # 5 minutes
    solver.options['MIPGap'] = 0.01    # 1% gap
    
    # Solve the model
    results = solver.solve(model, tee=True)  # tee=True shows solver output
    
    # 9. RESULT PROCESSING
    # Check solver status
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print("Optimal solution found!")
        print(f"Optimal value: {pyo.value(model.objective)}")
        
        # Extract variable values
        print("\nVariable values:")
        for i in model.I:
            x_val = pyo.value(model.x[i])
            if x_val > 1e-6:  # Only print non-zero values
                print(f"x[{i}] = {x_val:.3f}")
        
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")
    
    return model

# Execute the optimization
if __name__ == "__main__":
    manufactory_optimization()
```

### Execution Results
**Status**: OPTIMAL
**Optimal Value**: 24680.0
**Execution Time**: 0.92 seconds
**Reliability**: high
**Analysis**: Pyomo achieved the optimal solution with an execution time between Gurobipy and DOCplex, indicating moderate efficiency.

## 8. Cross-Solver Analysis and Final Recommendation

### Solver Results Comparison

| Solver | Status | Optimal Value | Execution Time | Decision Variables | Retry Attempt |
|--------|--------|---------------|----------------|-------------------|---------------|
| Gurobipy | OPTIMAL | 24680.00 | 0.18s | N/A | N/A |
| Docplex | OPTIMAL | 24680.00 | 1.11s | N/A | N/A |
| Pyomo | OPTIMAL | 24680.00 | 0.92s | N/A | N/A |

### Solver Consistency Analysis
**Result**: All solvers produced consistent results ✓
**Consistent Solvers**: gurobipy, docplex, pyomo
**Majority Vote Optimal Value**: 24680.0

### Final Recommendation
**Recommended Optimal Value**: 24680.0
**Confidence Level**: HIGH
**Preferred Solver(s)**: gurobipy
**Reasoning**: Gurobipy is recommended due to its optimal solution and significantly faster execution time compared to DOCplex and Pyomo.

### Business Interpretation
**Overall Strategy**: The optimal solution maximizes total revenue while adhering to budget, demand, and capacity constraints.
**Objective Value Meaning**: The optimal total revenue of 24680.0 indicates the maximum achievable revenue given the constraints.
**Resource Allocation Summary**: Resources should be allocated to produce the optimal number of units for each product, ensuring budget, demand, and capacity constraints are met.
**Implementation Recommendations**: Implement the production plan as per the optimal solution, ensuring continuous monitoring of budget and capacity to maintain optimal revenue.