#!/usr/bin/env python3
"""
Gurobipy 12.0.2 Implementation for Furniture Manufacturer Optimization Problem
"""

import gurobipy as gp
from gurobipy import GRB

def optimize_furniture_production():
    """Optimize the number of units to produce for each furniture type to maximize profit."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("furniture_production")
    
    # Data from the problem
    furniture_ids = [1, 2, 3]
    market_rates = {1: 120.0, 2: 180.0, 3: 250.0}
    production_costs = {1: 60.0, 2: 90.0, 3: 130.0}
    
    # Validate array lengths before loops
    assert len(furniture_ids) == len(market_rates) == len(production_costs), "Array length mismatch"
    
    # 2. VARIABLES
    x = {i: model.addVar(vtype=GRB.CONTINUOUS, name=f"x_{i}", lb=0) for i in furniture_ids}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize total profit: Z = sum((Market_Rate_i - Price_in_Dollar_i) * x_i)
    model.setObjective(
        gp.quicksum((market_rates[i] - production_costs[i]) * x[i] for i in furniture_ids),
        GRB.MAXIMIZE
    )
    
    # 4. CONSTRAINTS
    
    # Total Production Capacity: sum(x_i) <= 1000
    model.addConstr(
        gp.quicksum(x[i] for i in furniture_ids) <= 1000,
        name="total_production_capacity"
    )
    
    # Budget Limit: sum(Price_in_Dollar_i * x_i) <= 50000
    model.addConstr(
        gp.quicksum(production_costs[i] * x[i] for i in furniture_ids) <= 50000,
        name="budget_limit"
    )
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in furniture_ids:
            if x[i].x > 1e-6:
                print(f"x[{i}] = {x[i].x:.3f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Execute the optimization
if __name__ == "__main__":
    optimize_furniture_production()