# Complete DOCPLEX implementation - Retry Attempt 4

from docplex.mp.model import Model

def pilot_assignment_optimization():
    # 1. MODEL & DATA SETUP
    mdl = Model(name="pilot_assignment_optimization")
    
    # Data from the problem
    pilots = [1, 2, 3]
    aircraft = [101, 102, 103]
    
    cost_coefficients = {
        (1, 101): 2.55,
        (2, 102): 3.4,
        (3, 103): 4.25
    }
    
    pilot_qualifications = {
        (1, 101): 1,
        (2, 102): 1,
        (3, 103): 0
    }
    
    # CRITICAL: Validate array lengths to prevent IndexError
    assert all((p, a) in cost_coefficients for p in pilots for a in aircraft), "Missing cost coefficients"
    assert all((p, a) in pilot_qualifications for p in pilots for a in aircraft), "Missing pilot qualifications"
    
    # 2. VARIABLES
    x = {(p, a): mdl.binary_var(name=f"x_{p}_{a}") for p in pilots for a in aircraft}
    
    # 3. OBJECTIVE FUNCTION
    objective = mdl.sum(cost_coefficients[(p, a)] * x[(p, a)] for p in pilots for a in aircraft)
    mdl.minimize(objective)
    
    # 4. CONSTRAINTS
    
    # Pilot Assignment Limit: Each pilot can be assigned to at most one aircraft
    for p in pilots:
        mdl.add_constraint(mdl.sum(x[(p, a)] for a in aircraft) <= 1, ctname=f"pilot_limit_{p}")
    
    # Aircraft Assignment Requirement: Each aircraft must be assigned exactly one pilot
    for a in aircraft:
        mdl.add_constraint(mdl.sum(x[(p, a)] for p in pilots) == 1, ctname=f"aircraft_req_{a}")
    
    # Pilot Qualifications: A pilot can only be assigned to an aircraft if they are qualified
    for p in pilots:
        for a in aircraft:
            mdl.add_constraint(x[(p, a)] <= pilot_qualifications[(p, a)], ctname=f"qualification_{p}_{a}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for p in pilots:
            for a in aircraft:
                if solution.get_value(x[(p, a)]) > 0:
                    print(f"Pilot {p} is assigned to Aircraft {a}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
    
    return mdl

# Execute the optimization
pilot_assignment_optimization()