# Complete Optimization Problem and Solution: program_share

## 1. Problem Context and Goals

### Context  
A media company is focused on maximizing its total viewership share across multiple channels by strategically scheduling programs. The company must decide which programs to broadcast on which channels and at what times of day. This decision is guided by two key operational metrics: the share of viewership each program attracts on a specific channel and the overall rating of the channel itself. The goal is to ensure that the scheduling decisions are made in a way that maximizes the combined impact of these metrics.

The company operates under specific business rules:  
1. Each program can be broadcast at most once across all channels and time slots.  
2. Each channel has a maximum number of time slots available for broadcasting, currently set to 5.  
3. Each channel must broadcast a minimum number of local programs, currently set to 2, to comply with regulatory requirements and cater to local audiences.  

These rules ensure that the scheduling process remains balanced and adheres to both operational and regulatory constraints. The decision variables are binary, indicating whether a specific program is scheduled on a particular channel at a given time slot. The objective is to maximize the total viewership share, calculated as the sum of the products of program shares and channel ratings for all scheduled programs.

### Goals  
The primary goal of this optimization problem is to maximize the total viewership share across all channels. This is achieved by selecting the best combination of programs, channels, and time slots, weighted by the program's share of viewership on the channel and the channel's overall rating. Success is measured by the total viewership share generated from the scheduled programs, ensuring that the company's content reaches the largest possible audience while adhering to operational and regulatory constraints.

## 2. Constraints  

The optimization problem is subject to the following constraints:  
1. **Program Uniqueness**: Each program can be broadcast at most once across all channels and time slots. This ensures that no program is over-scheduled, maintaining variety in the programming lineup.  
2. **Channel Time Slot Limit**: Each channel can only use a limited number of time slots for broadcasting, currently capped at 5. This ensures that no channel is overburdened with too many programs, maintaining a balanced schedule.  
3. **Local Program Requirement**: Each channel must broadcast a minimum number of local programs, currently set to 2. This ensures compliance with regulatory requirements and caters to the preferences of local audiences.  

These constraints are designed to ensure that the scheduling decisions are both operationally feasible and aligned with the company's strategic and regulatory goals.

## 3. Available Data  

### Database Schema  
```sql
-- Iteration 3 Database Schema
-- Objective: Schema changes include adding tables for Max_Time_Slots and Min_Local_Programs to address missing optimization requirements. Configuration logic updates include scalar parameters for Max_Time_Slots and Min_Local_Programs.

CREATE TABLE time_slots (
  time_slot_id INTEGER,
  time_of_day STRING
);

CREATE TABLE program_origins (
  program_id INTEGER,
  origin STRING
);

CREATE TABLE broadcast_decisions (
  channel_id INTEGER,
  program_id INTEGER,
  time_slot_id INTEGER,
  x BOOLEAN
);

CREATE TABLE program_shares (
  channel_id INTEGER,
  program_id INTEGER,
  share_in_percent INTEGER
);

CREATE TABLE channel_ratings (
  channel_id INTEGER,
  rating_in_percent INTEGER
);
```

### Data Dictionary  
- **time_slots**: Contains available time slots for broadcasting programs.  
  - `time_slot_id`: Unique identifier for a time slot.  
  - `time_of_day`: Time of day for broadcasting (e.g., Morning, Afternoon, Evening).  

- **program_origins**: Indicates the origin of programs (e.g., local, international).  
  - `program_id`: Unique identifier for a program.  
  - `origin`: Origin of the program (e.g., Local, International).  

- **broadcast_decisions**: Represents binary decisions indicating whether a program is broadcast on a channel at a specific time.  
  - `channel_id`: Unique identifier for a channel.  
  - `program_id`: Unique identifier for a program.  
  - `time_slot_id`: Unique identifier for a time slot.  
  - `x`: Binary decision variable (0 or 1).  

- **program_shares**: Contains the share of viewership for a program on a channel.  
  - `channel_id`: Unique identifier for a channel.  
  - `program_id`: Unique identifier for a program.  
  - `share_in_percent`: Share of viewership for the program on the channel.  

- **channel_ratings**: Contains the rating of a channel.  
  - `channel_id`: Unique identifier for a channel.  
  - `rating_in_percent`: Rating of the channel.  

### Current Stored Values  
```sql
-- Iteration 3 Realistic Data
-- Generated by triple expert (business + data + optimization)
-- Values were determined based on typical media industry standards, ensuring realistic viewership shares, channel ratings, and program scheduling constraints. The data was designed to reflect a balanced mix of local and international programs, with varying time slots to maximize viewership.

-- Realistic data for time_slots
INSERT INTO time_slots (time_slot_id, time_of_day) VALUES (1, 'Morning');
INSERT INTO time_slots (time_slot_id, time_of_day) VALUES (2, 'Afternoon');
INSERT INTO time_slots (time_slot_id, time_of_day) VALUES (3, 'Evening');

-- Realistic data for program_origins
INSERT INTO program_origins (program_id, origin) VALUES (1, 'Local');
INSERT INTO program_origins (program_id, origin) VALUES (2, 'International');
INSERT INTO program_origins (program_id, origin) VALUES (3, 'Local');

-- Realistic data for broadcast_decisions
INSERT INTO broadcast_decisions (channel_id, program_id, time_slot_id, x) VALUES (1, 1, 1, 1);
INSERT INTO broadcast_decisions (channel_id, program_id, time_slot_id, x) VALUES (2, 2, 3, 1);
INSERT INTO broadcast_decisions (channel_id, program_id, time_slot_id, x) VALUES (3, 3, 2, 1);

-- Realistic data for program_shares
INSERT INTO program_shares (channel_id, program_id, share_in_percent) VALUES (1, 1, 60);
INSERT INTO program_shares (channel_id, program_id, share_in_percent) VALUES (2, 2, 70);
INSERT INTO program_shares (channel_id, program_id, share_in_percent) VALUES (3, 3, 50);

-- Realistic data for channel_ratings
INSERT INTO channel_ratings (channel_id, rating_in_percent) VALUES (1, 75);
INSERT INTO channel_ratings (channel_id, rating_in_percent) VALUES (2, 85);
INSERT INTO channel_ratings (channel_id, rating_in_percent) VALUES (3, 80);
```

## 4. Mathematical Optimization Formulation

#### Decision Variables
Let \( x_{c,p,t} \) be a binary decision variable where:  
- \( x_{c,p,t} = 1 \) if program \( p \) is scheduled on channel \( c \) at time slot \( t \),  
- \( x_{c,p,t} = 0 \) otherwise.  

#### Objective Function
Maximize the total viewership share:  
\[
\text{Maximize } \sum_{c} \sum_{p} \sum_{t} (\text{program\_shares.share\_in\_percent}_{c,p} \times \text{channel\_ratings.rating\_in\_percent}_{c}) \times x_{c,p,t}
\]  
Data Source Verification:  
- \( \text{program\_shares.share\_in\_percent}_{c,p} \) comes from `program_shares.share_in_percent`.  
- \( \text{channel\_ratings.rating\_in\_percent}_{c} \) comes from `channel_ratings.rating_in_percent`.  

#### Constraints
1. **Program Uniqueness**: Each program can be broadcast at most once across all channels and time slots:  
\[
\sum_{c} \sum_{t} x_{c,p,t} \leq 1 \quad \forall p
\]  
Data Source Verification: This constraint ensures no program is over-scheduled.  

2. **Channel Time Slot Limit**: Each channel can use at most 5 time slots:  
\[
\sum_{p} \sum_{t} x_{c,p,t} \leq 5 \quad \forall c
\]  
Data Source Verification: This constraint ensures no channel exceeds its time slot limit.  

3. **Local Program Requirement**: Each channel must broadcast at least 2 local programs:  
\[
\sum_{p \in \text{Local}} \sum_{t} x_{c,p,t} \geq 2 \quad \forall c
\]  
Data Source Verification: \( p \in \text{Local} \) is determined by `program_origins.origin = 'Local'`.  

4. **Binary Decision Variables**:  
\[
x_{c,p,t} \in \{0, 1\} \quad \forall c, p, t
\]  
Data Source Verification: This ensures \( x_{c,p,t} \) is binary.  

This is a complete, immediately solvable Mixed-Integer Linear Programming (MIP) model with all numerical coefficients derived from the provided data.

## 5. Gurobipy Implementation

```python
# Complete GUROBIPY implementation - Retry Attempt 4

import gurobipy as gp
from gurobipy import GRB

def program_share_optimization():
    # 1. MODEL & DATA SETUP
    model = gp.Model("program_share_optimization")
    
    # Data from the database
    channels = [1, 2, 3]
    programs = [1, 2, 3]
    time_slots = [1, 2, 3]
    
    # Program shares and channel ratings
    program_shares = {
        (1, 1): 60,
        (2, 2): 70,
        (3, 3): 50
    }
    
    channel_ratings = {
        1: 75,
        2: 85,
        3: 80
    }
    
    # Program origins
    program_origins = {
        1: 'Local',
        2: 'International',
        3: 'Local'
    }
    
    # Validate array lengths before loops
    assert len(program_shares) == len(programs) * len(channels), "Program shares length mismatch"
    assert len(channel_ratings) == len(channels), "Channel ratings length mismatch"
    assert len(program_origins) == len(programs), "Program origins length mismatch"
    
    # 2. VARIABLES
    x = model.addVars(channels, programs, time_slots, vtype=GRB.BINARY, name="x")
    
    # 3. OBJECTIVE FUNCTION
    model.setObjective(
        gp.quicksum(
            program_shares[c, p] * channel_ratings[c] * x[c, p, t]
            for c in channels
            for p in programs
            for t in time_slots
        ),
        GRB.MAXIMIZE
    )
    
    # 4. CONSTRAINTS
    
    # Program Uniqueness: Each program can be broadcast at most once across all channels and time slots
    for p in programs:
        model.addConstr(
            gp.quicksum(x[c, p, t] for c in channels for t in time_slots) <= 1,
            name=f"program_uniqueness_{p}"
        )
    
    # Channel Time Slot Limit: Each channel can use at most 5 time slots
    for c in channels:
        model.addConstr(
            gp.quicksum(x[c, p, t] for p in programs for t in time_slots) <= 5,
            name=f"channel_time_slot_limit_{c}"
        )
    
    # Local Program Requirement: Each channel must broadcast at least 2 local programs
    for c in channels:
        model.addConstr(
            gp.quicksum(x[c, p, t] for p in programs if program_origins[p] == 'Local' for t in time_slots) >= 2,
            name=f"local_program_requirement_{c}"
        )
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for c in channels:
            for p in programs:
                for t in time_slots:
                    if x[c, p, t].x > 0.5:
                        print(f"Channel {c} broadcasts Program {p} at Time Slot {t}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Execute the optimization
program_share_optimization()
```

### Execution Results
**Status**: ERROR
**Error**: Traceback (most recent call last):
  File "/tmp/tmpv2n5llqq.py", line 95, in <module>
    program_share_optimization()
  File "/tmp/tmpv2n5llqq.py", line 36, in program_share_optimization
    assert len(program_shares) == len(programs) * len(channels), "Program shares length mismatch"
AssertionError: Program shares length mismatch

**Analysis**: Gurobipy failed due to a data mismatch error. The error message indicates that the length of `program_shares` does not match the expected length based on the number of programs and channels. This suggests a potential issue with data preparation or input validation.

## 6. DOCplex Implementation

```python
# Complete DOCPLEX implementation - Retry Attempt 4

from docplex.mp.model import Model

def program_share_optimization():
    # 1. MODEL & DATA SETUP
    mdl = Model(name="program_share_optimization")
    
    # Data from the database
    channels = [1, 2, 3]
    programs = [1, 2, 3]
    time_slots = [1, 2, 3]
    
    program_shares = {
        (1, 1): 60,
        (2, 2): 70,
        (3, 3): 50
    }
    
    channel_ratings = {
        1: 75,
        2: 85,
        3: 80
    }
    
    program_origins = {
        1: 'Local',
        2: 'International',
        3: 'Local'
    }
    
    # CRITICAL: Validate array lengths to prevent IndexError
    assert len(program_shares) == len(programs) * len(channels), "Program shares length mismatch"
    
    # 2. VARIABLES
    x = {(c, p, t): mdl.binary_var(name=f"x_{c}_{p}_{t}") 
         for c in channels 
         for p in programs 
         for t in time_slots}
    
    # 3. OBJECTIVE FUNCTION
    objective = mdl.sum(program_shares[(c, p)] * channel_ratings[c] * x[(c, p, t)] 
                        for c in channels 
                        for p in programs 
                        for t in time_slots)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS
    
    # Program Uniqueness: Each program can be broadcast at most once
    for p in programs:
        mdl.add_constraint(mdl.sum(x[(c, p, t)] for c in channels for t in time_slots) <= 1, 
                           ctname=f"program_uniqueness_{p}")
    
    # Channel Time Slot Limit: Each channel can use at most 5 time slots
    for c in channels:
        mdl.add_constraint(mdl.sum(x[(c, p, t)] for p in programs for t in time_slots) <= 5, 
                           ctname=f"time_slot_limit_{c}")
    
    # Local Program Requirement: Each channel must broadcast at least 2 local programs
    for c in channels:
        mdl.add_constraint(mdl.sum(x[(c, p, t)] for p in programs if program_origins[p] == 'Local' for t in time_slots) >= 2, 
                           ctname=f"local_program_requirement_{c}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for c in channels:
            for p in programs:
                for t in time_slots:
                    if solution.get_value(x[(c, p, t)]) > 0.5:
                        print(f"Channel {c} broadcasts Program {p} at Time Slot {t}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
    
    return mdl

# Execute the optimization
program_share_optimization()
```

### Execution Results
**Status**: ERROR
**Error**: Traceback (most recent call last):
  File "/tmp/tmpcs8bll6k.py", line 82, in <module>
    program_share_optimization()
  File "/tmp/tmpcs8bll6k.py", line 33, in program_share_optimization
    assert len(program_shares) == len(programs) * len(channels), "Program shares length mismatch"
AssertionError: Program shares length mismatch

**Analysis**: DOCplex encountered the same data mismatch error as Gurobipy. The error message confirms that the length of `program_shares` is inconsistent with the expected dimensions, indicating a systemic issue with the data rather than the solver itself.

## 7. Pyomo Implementation

```python
# Complete PYOMO implementation - Retry Attempt 4

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def program_share_optimization():
    # 1. MODEL CREATION
    model = pyo.ConcreteModel()
    
    # 2. DATA SETUP
    # Define sets
    channels = [1, 2, 3]
    programs = [1, 2, 3]
    time_slots = [1, 2, 3]
    
    # Define parameters
    program_shares = {
        (1, 1): 60,
        (2, 2): 70,
        (3, 3): 50
    }
    
    channel_ratings = {
        1: 75,
        2: 85,
        3: 80
    }
    
    program_origins = {
        1: 'Local',
        2: 'International',
        3: 'Local'
    }
    
    # Validate array lengths
    assert len(program_shares) == len(programs) * len(channels), "Program shares length mismatch"
    assert len(channel_ratings) == len(channels), "Channel ratings length mismatch"
    assert len(program_origins) == len(programs), "Program origins length mismatch"
    
    # 3. SETS
    model.C = pyo.Set(initialize=channels)
    model.P = pyo.Set(initialize=programs)
    model.T = pyo.Set(initialize=time_slots)
    
    # 4. PARAMETERS
    model.program_shares = pyo.Param(model.C, model.P, initialize=program_shares)
    model.channel_ratings = pyo.Param(model.C, initialize=channel_ratings)
    model.program_origins = pyo.Param(model.P, initialize=program_origins)
    
    # 5. VARIABLES
    model.x = pyo.Var(model.C, model.P, model.T, within=pyo.Binary)
    
    # 6. OBJECTIVE FUNCTION
    def obj_rule(model):
        return sum(model.program_shares[c, p] * model.channel_ratings[c] * model.x[c, p, t] for c in model.C for p in model.P for t in model.T)
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.maximize)
    
    # 7. CONSTRAINTS
    # Program Uniqueness
    def program_uniqueness_rule(model, p):
        return sum(model.x[c, p, t] for c in model.C for t in model.T) <= 1
    model.program_uniqueness = pyo.Constraint(model.P, rule=program_uniqueness_rule)
    
    # Channel Time Slot Limit
    def channel_time_slot_limit_rule(model, c):
        return sum(model.x[c, p, t] for p in model.P for t in model.T) <= 5
    model.channel_time_slot_limit = pyo.Constraint(model.C, rule=channel_time_slot_limit_rule)
    
    # Local Program Requirement
    def local_program_requirement_rule(model, c):
        return sum(model.x[c, p, t] for p in model.P if model.program_origins[p] == 'Local' for t in model.T) >= 2
    model.local_program_requirement = pyo.Constraint(model.C, rule=local_program_requirement_rule)
    
    # 8. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')
    
    # Optional: Set solver options
    solver.options['TimeLimit'] = 300  # 5 minutes
    solver.options['MIPGap'] = 0.01    # 1% gap
    
    # Solve the model
    results = solver.solve(model, tee=True)  # tee=True shows solver output
    
    # 9. RESULT PROCESSING
    # Check solver status
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print("Optimal solution found!")
        print(f"Optimal value: {pyo.value(model.objective)}")
        
        # Extract variable values
        print("\nVariable values:")
        for c in model.C:
            for p in model.P:
                for t in model.T:
                    if pyo.value(model.x[c, p, t]) > 0:
                        print(f"x[{c}, {p}, {t}] = {pyo.value(model.x[c, p, t])}")
        
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")
    
    return model

# Execute the optimization
if __name__ == "__main__":
    program_share_optimization()
```

### Execution Results
**Status**: ERROR
**Error**: Traceback (most recent call last):
  File "/tmp/tmp1lgdwj7z.py", line 109, in <module>
    program_share_optimization()
  File "/tmp/tmp1lgdwj7z.py", line 36, in program_share_optimization
    assert len(program_shares) == len(programs) * len(channels), "Program shares length mismatch"
AssertionError: Program shares length mismatch

**Analysis**: Pyomo also failed with the same data mismatch error. This consistency across all solvers strongly suggests that the issue lies in the data preparation or input validation process, not in the solvers.

## 8. Cross-Solver Analysis and Final Recommendation

### Solver Results Comparison

| Solver | Status | Optimal Value | Execution Time | Decision Variables | Retry Attempt |
|--------|--------|---------------|----------------|-------------------|---------------|
| Gurobipy | ERROR | N/A | 0.43s | N/A | 4 |
| Docplex | ERROR | N/A | 5.85s | N/A | 4 |
| Pyomo | ERROR | N/A | 1.23s | N/A | 4 |

### Solver Consistency Analysis
**Result**: Solvers produced inconsistent results
**Inconsistent Solvers**: gurobipy, docplex, pyomo
**Potential Issues**:
- Data preparation error: The length of `program_shares` does not match the expected dimensions.
- Input validation failure: The data may not have been properly checked before being passed to the solvers.
- Potential missing or incorrect data in the `program_shares` dataset.
**Solver Retry Summary**: gurobipy: 4 attempts, docplex: 4 attempts, pyomo: 4 attempts

### Final Recommendation
**Confidence Level**: LOW
**Preferred Solver(s)**: none
**Reasoning**: All solvers failed due to the same data mismatch error. The issue must be resolved in the data preparation or input validation stage before any solver can be used effectively.

### Business Interpretation
**Overall Strategy**: The optimization process failed due to a data mismatch, meaning no valid schedule could be generated. This could lead to suboptimal program scheduling and potential loss of viewership.
**Objective Value Meaning**: Not applicable due to solver failure.
**Resource Allocation Summary**: Not applicable due to solver failure.
**Implementation Recommendations**: 1. Verify the `program_shares` dataset to ensure it matches the expected dimensions. 2. Implement robust input validation to catch data mismatches before running the optimization. 3. Re-run the optimization with corrected data to generate a valid schedule.