#!/usr/bin/env python3
"""
Gurobipy 12.0.2 Implementation for School Finance Optimization Problem
"""

import gurobipy as gp
from gurobipy import GRB

def school_finance_optimization():
    """Optimize school budget allocation to maximize educational impact."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("school_finance")
    
    # Data from the problem
    schools = [1, 2, 3]
    weights = {1: 0.6, 2: 0.8, 3: 0.5}
    min_investments = {1: 60000, 2: 70000, 3: 50000}
    total_budget = 1000000
    
    # CRITICAL: Validate array lengths before loops
    assert len(schools) == len(weights) == len(min_investments), "Array length mismatch"
    
    # 2. VARIABLES
    x = {i: model.addVar(vtype=GRB.CONTINUOUS, name=f"x_{i}", lb=0) for i in schools}
    
    # 3. OBJECTIVE FUNCTION
    model.setObjective(gp.quicksum(weights[i] * x[i] for i in schools), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS
    
    # Total Budget Constraint
    model.addConstr(gp.quicksum(x[i] for i in schools) <= total_budget, name="total_budget")
    
    # Minimum Investment Constraints
    for i in schools:
        model.addConstr(x[i] >= min_investments[i], name=f"min_investment_{i}")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in schools:
            print(f"Investment in School {i}: {x[i].x:.2f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Execute the optimization
if __name__ == "__main__":
    school_finance_optimization()