#!/usr/bin/env python3
"""
DOCPLEX implementation for the voter_1 optimization problem
"""

from docplex.mp.model import Model

def voter_optimization():
    """Optimize promotional budget allocation to maximize votes"""
    
    # 1. MODEL & DATA SETUP
    mdl = Model(name="voter_optimization")
    
    # Contestants and states
    contestants = [1, 2, 3]
    states = ['CA', 'TX', 'NY']
    
    # Parameters
    total_budget = 150000
    max_budget_per_state = 25000
    conversion_rate = 0.6
    
    # 2. VARIABLES
    # Decision variable: promotional budget allocated to each contestant in each state
    x = {(i, j): mdl.continuous_var(name=f"x_{i}_{j}", lb=0) for i in contestants for j in states}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize total votes received across all contestants and states
    objective = mdl.sum(conversion_rate * x[i, j] for i in contestants for j in states)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS
    
    # Total Budget Constraint
    total_budget_expr = mdl.sum(x[i, j] for i in contestants for j in states)
    mdl.add_constraint(total_budget_expr <= total_budget, ctname="total_budget")
    
    # State-Level Budget Constraint
    for i in contestants:
        for j in states:
            mdl.add_constraint(x[i, j] <= max_budget_per_state, ctname=f"state_budget_{i}_{j}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in contestants:
            for j in states:
                value = solution.get_value(x[i, j])
                if value > 1e-6:
                    print(f"x[{i}, {j}] = {value:.3f}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
    
    return mdl

# Execute the optimization
if __name__ == "__main__":
    voter_optimization()