#!/usr/bin/env python3
"""
Gurobipy 12.0.2 Implementation for Wine Production Optimization
"""

import gurobipy as gp
from gurobipy import GRB

def wine_production_optimization():
    """Optimize wine production to maximize profit."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("wine_production")
    
    # Data from the problem
    profits = [10.5, 15.0, 20.0]
    grape_usages = [2.5, 3.0, 4.0]
    production_costs = [5.0, 6.0, 7.0]
    market_demands = [100, 150, 200]
    total_budget = 100000
    total_grapes_available = 1000  # Assuming a value since not provided
    
    n_wines = len(profits)
    
    # CRITICAL: Validate array lengths before loops
    assert len(profits) == len(grape_usages) == len(production_costs) == len(market_demands) == n_wines, "Array length mismatch"
    
    # 2. VARIABLES
    x = {i: model.addVar(vtype=GRB.INTEGER, name=f"x_{i}", lb=0) for i in range(n_wines)}
    
    # 3. OBJECTIVE FUNCTION
    model.setObjective(gp.quicksum(profits[i] * x[i] for i in range(n_wines)), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS
    
    # Grape availability constraint
    model.addConstr(gp.quicksum(grape_usages[i] * x[i] for i in range(n_wines)) <= total_grapes_available, name="grape_availability")
    
    # Production budget constraint
    model.addConstr(gp.quicksum(production_costs[i] * x[i] for i in range(n_wines)) <= total_budget, name="budget_constraint")
    
    # Market demand constraints
    for i in range(n_wines):
        model.addConstr(x[i] <= market_demands[i], name=f"demand_{i}")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in range(n_wines):
            print(f"x[{i}] = {x[i].x}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Execute the optimization
if __name__ == "__main__":
    wine_production_optimization()