#!/usr/bin/env python3
"""
Gurobipy implementation for workshop paper selection optimization
"""

import gurobipy as gp
from gurobipy import GRB

def optimize_workshop_selection():
    """Optimize the selection of workshop submissions to maximize quality scores."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("workshop_paper_selection")
    
    # Data from the problem
    workshop_capacity = {1: 12, 2: 15, 3: 10}
    submission_scores = {1: 8.7, 2: 7.5, 3: 9.2}
    submission_workshop_mapping = {1: 1, 2: 2, 3: 3}
    
    # Validate array lengths
    assert len(submission_scores) == len(submission_workshop_mapping), "Array length mismatch"
    
    # 2. VARIABLES
    # Binary decision variables for each submission
    x = {i: model.addVar(vtype=GRB.BINARY, name=f"x_{i}") for i in submission_scores.keys()}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the total quality score of accepted submissions
    model.setObjective(gp.quicksum(submission_scores[i] * x[i] for i in submission_scores.keys()), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS
    
    # Workshop Capacity Constraints
    for workshop_id, capacity in workshop_capacity.items():
        # Get submissions assigned to this workshop
        submissions_in_workshop = [i for i, w_id in submission_workshop_mapping.items() if w_id == workshop_id]
        model.addConstr(gp.quicksum(x[i] for i in submissions_in_workshop) <= capacity, name=f"capacity_{workshop_id}")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in submission_scores.keys():
            if x[i].x > 0.5:
                print(f"Submission {i} is accepted")
            else:
                print(f"Submission {i} is rejected")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
if __name__ == "__main__":
    optimize_workshop_selection()