# Complete DOCPLEX implementation - Retry Attempt 4

from docplex.mp.model import Model

def optimize_tournament_scheduling():
    # 1. MODEL & DATA SETUP
    mdl = Model(name="tournament_scheduling")
    
    # Data from the problem
    players = [1, 2, 3]
    tournaments = [101, 102, 103]
    
    # Ranking points data
    ranking_points = {
        (1, 101): 100,
        (2, 102): 200,
        (3, 103): 300
    }
    
    # Player availability data
    availability = {
        1: True,
        2: False,
        3: True
    }
    
    # Tournament capacity data
    max_players_per_tournament = {
        101: 32,
        102: 64,
        103: 128
    }
    
    # 2. VARIABLES
    x = mdl.binary_var_dict([(p, t) for p in players for t in tournaments], name="x")
    
    # 3. OBJECTIVE FUNCTION
    objective = mdl.sum(ranking_points.get((p, t), 0) * x[p, t] for p in players for t in tournaments)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS
    
    # Player Participation Limit
    for p in players:
        mdl.add_constraint(mdl.sum(x[p, t] for t in tournaments) <= 5, ctname=f"participation_limit_{p}")
    
    # Tournament Capacity Limit
    for t in tournaments:
        mdl.add_constraint(mdl.sum(x[p, t] for p in players) <= max_players_per_tournament[t], ctname=f"capacity_limit_{t}")
    
    # Player Availability
    for p in players:
        for t in tournaments:
            mdl.add_constraint(x[p, t] <= int(availability[p]), ctname=f"availability_{p}_{t}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for p in players:
            for t in tournaments:
                if solution.get_value(x[p, t]) > 0:
                    print(f"Player {p} participates in tournament {t}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
    
    return mdl

# Execute the optimization
optimize_tournament_scheduling()