# Complete DOCPLEX implementation

from docplex.mp.model import Model

def optimize_aircraft_allocation():
    """Optimize aircraft allocation to maximize passenger transport"""
    
    # 1. MODEL & DATA SETUP
    mdl = Model(name="aircraft_allocation")
    
    # Data: Passenger capacities and aircraft assignments
    passenger_capacities = [150, 180, 200]  # Example capacities for aircraft types
    aircraft_assignments = [3, 5, 7]  # Example number of aircraft assigned to airports
    movement_limits = [10, 12, 15]  # Example movement limits for airports
    
    # CRITICAL: Validate array lengths to prevent IndexError
    assert len(passenger_capacities) == len(aircraft_assignments) == len(movement_limits), "Array length mismatch"
    safe_range = range(min(len(passenger_capacities), len(aircraft_assignments), len(movement_limits)))  # Safe indexing
    
    # 2. VARIABLES
    # Decision variables: number of aircraft of type i assigned to airport j
    x = {(i, j): mdl.integer_var(name=f"x_{i}_{j}", lb=0) for i in safe_range for j in safe_range}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize total number of passengers transported
    objective = mdl.sum(passenger_capacities[i] * x[i, j] for i in safe_range for j in safe_range)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Aircraft Capacity Constraint
    for i in safe_range:
        mdl.add_constraint(mdl.sum(x[i, j] for j in safe_range) <= aircraft_assignments[i], ctname=f"aircraft_capacity_{i}")
    
    # Airport Movement Constraint
    for j in safe_range:
        mdl.add_constraint(mdl.sum(x[i, j] for i in safe_range) <= movement_limits[j], ctname=f"airport_movement_{j}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in safe_range:
            for j in safe_range:
                value = solution.get_value(x[i, j])
                if value > 1e-6:
                    print(f"x[{i},{j}] = {value:.3f}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
    
    return mdl

# Run the optimization
optimize_aircraft_allocation()