# Complete GUROBIPY implementation

import gurobipy as gp
from gurobipy import GRB

def optimize_book_publication():
    """Optimize the number of issues to publish for each book to maximize revenue."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("book_publication")
    
    # Prices from the publication table
    prices = [18.99, 22.5, 14.75]
    
    # Maximum number of issues allowed and total budget available
    M = 50  # Example value for maximum number of issues
    B = 1000  # Example value for total budget
    
    # Number of books
    n_books = len(prices)
    
    # CRITICAL: Validate array lengths before loops
    assert len(prices) == n_books, "Array length mismatch"
    
    # 2. VARIABLES
    # Decision variables for the number of issues to publish for each book
    issues = model.addVars(n_books, vtype=GRB.INTEGER, name="issues", lb=0)
    
    # 3. OBJECTIVE FUNCTION
    # Maximize total revenue
    model.setObjective(gp.quicksum(prices[i] * issues[i] for i in range(n_books)), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Total number of issues constraint
    model.addConstr(gp.quicksum(issues[i] for i in range(n_books)) <= M, name="total_issues_limit")
    
    # Budget constraint
    model.addConstr(gp.quicksum(prices[i] * issues[i] for i in range(n_books)) <= B, name="budget_limit")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in range(n_books):
            if issues[i].x > 1e-6:
                print(f"issues[{i}] = {issues[i].x:.0f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
optimize_book_publication()