# Complete GUROBIPY implementation

import gurobipy as gp
from gurobipy import GRB

def climbing_optimization():
    """Optimize climber assignments to mountains to maximize points."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("climbing_competition")
    
    # Data: Points each climber can score
    climber_points = [15, 25, 10]
    n_climbers = len(climber_points)
    
    # Data: Number of mountains
    n_mountains = 3
    
    # CRITICAL: Validate array lengths before loops
    assert n_climbers == 3, "Unexpected number of climbers"
    assert n_mountains == 3, "Unexpected number of mountains"
    
    # 2. VARIABLES
    # Decision variables: x[i, j] = 1 if climber i is assigned to mountain j
    x = model.addVars(n_climbers, n_mountains, vtype=GRB.BINARY, name="x")
    
    # 3. OBJECTIVE FUNCTION
    # Maximize total points scored by all climbers assigned to mountains
    model.setObjective(gp.quicksum(climber_points[i] * x[i, j] 
                                   for i in range(n_climbers) 
                                   for j in range(n_mountains)), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Each climber must be assigned to exactly one mountain
    model.addConstrs((gp.quicksum(x[i, j] for j in range(n_mountains)) == 1 
                      for i in range(n_climbers)), name="climber_assignment")
    
    # Each mountain must have at least one climber assigned to it
    model.addConstrs((gp.quicksum(x[i, j] for i in range(n_climbers)) >= 1 
                      for j in range(n_mountains)), name="mountain_assignment")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in range(n_climbers):
            for j in range(n_mountains):
                if x[i, j].x > 1e-6:
                    print(f"Climber {i+1} assigned to Mountain {j+1}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
climbing_optimization()