# Complete DOCPLEX implementation

from docplex.mp.model import Model

def optimize_staff_allocation():
    """Optimize staff allocation for coffee shop happy hours"""
    
    # 1. MODEL & DATA SETUP
    mdl = Model(name="coffee_shop")
    
    # Data from the problem context
    shop_ids = [1, 2, 3]
    hh_ids = [101, 102, 103]
    scores = [4.5, 3.8, 4.2]
    max_staff = [5, 8, 6]
    total_staff_available = 19
    min_staff_per_hh = 2
    
    # CRITICAL: Validate array lengths to prevent IndexError
    assert len(shop_ids) == len(hh_ids) == len(scores) == len(max_staff), "Array length mismatch"
    safe_range = range(len(shop_ids))  # Safe indexing
    
    # 2. VARIABLES
    # Integer variables for staff allocation
    x = {i: mdl.integer_var(name=f"x_{hh_ids[i]}", lb=0) for i in safe_range}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the overall customer satisfaction score
    objective = mdl.sum(scores[i] * x[i] for i in safe_range)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Total staff allocation constraint
    total_allocation = mdl.sum(x[i] for i in safe_range)
    mdl.add_constraint(total_allocation <= total_staff_available, ctname="total_staff_limit")
    
    # Minimum staff requirement for each happy hour event
    for i in safe_range:
        mdl.add_constraint(x[i] >= min_staff_per_hh, ctname=f"min_staff_{hh_ids[i]}")
    
    # Maximum staff available at each shop
    for i in safe_range:
        mdl.add_constraint(x[i] <= max_staff[i], ctname=f"max_staff_{hh_ids[i]}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in safe_range:
            value = solution.get_value(x[i])
            print(f"x[{hh_ids[i]}] = {value:.0f}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
    
    return mdl

# Run the optimization
optimize_staff_allocation()