# Complete PYOMO implementation

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def coffee_shop_optimization():
    """Optimize staff allocation for coffee shop happy hours"""
    
    # 1. MODEL CREATION
    model = pyo.ConcreteModel()
    
    # 2. DATA SETUP
    # Happy hour event IDs
    hh_ids = [101, 102, 103]
    # Satisfaction scores for each shop
    scores = {101: 4.5, 102: 3.8, 103: 4.2}
    # Maximum staff available at each shop
    max_staff = {101: 5, 102: 8, 103: 6}
    # Total staff available across all shops
    total_staff_available = 19
    # Minimum staff required per happy hour
    min_staff_required = 2
    
    # CRITICAL: Validate data lengths before indexing
    assert len(hh_ids) == len(scores) == len(max_staff), "Data length mismatch"
    
    # 3. SETS
    model.HH = pyo.Set(initialize=hh_ids)
    
    # 4. PARAMETERS
    model.score = pyo.Param(model.HH, initialize=scores)
    model.max_staff = pyo.Param(model.HH, initialize=max_staff)
    
    # 5. VARIABLES
    model.x = pyo.Var(model.HH, within=pyo.NonNegativeIntegers)
    
    # 6. OBJECTIVE FUNCTION
    def obj_rule(model):
        return sum(model.score[i] * model.x[i] for i in model.HH)
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.maximize)
    
    # 7. CONSTRAINTS
    
    # Total staff allocation constraint
    def total_staff_rule(model):
        return sum(model.x[i] for i in model.HH) <= total_staff_available
    model.total_staff_constraint = pyo.Constraint(rule=total_staff_rule)
    
    # Minimum staff requirement for each happy hour event
    def min_staff_rule(model, i):
        return model.x[i] >= min_staff_required
    model.min_staff_constraint = pyo.Constraint(model.HH, rule=min_staff_rule)
    
    # Maximum staff available at each shop
    def max_staff_rule(model, i):
        return model.x[i] <= model.max_staff[i]
    model.max_staff_constraint = pyo.Constraint(model.HH, rule=max_staff_rule)
    
    # 8. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')
    
    # Solve the model
    results = solver.solve(model, tee=True)
    
    # 9. RESULT PROCESSING
    # Check solver status
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print("Optimal solution found!")
        print(f"Optimal value: {pyo.value(model.objective)}")
        
        # Extract variable values
        print("\nStaff allocation:")
        for i in model.HH:
            x_val = pyo.value(model.x[i])
            if x_val > 1e-6:  # Only print non-zero values
                print(f"Staff for HH_ID {i} = {int(x_val)}")
        
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")
    
    return model

# Run the optimization
coffee_shop_optimization()