# Complete DOCPLEX implementation

from docplex.mp.model import Model

def optimize_employee_allocation():
    """Optimize employee allocation to maximize profits"""
    
    # 1. MODEL & DATA SETUP
    mdl = Model(name="company_employee")
    
    # Data from the problem statement
    profits = [1.5, 2.0, 3.0]  # Profits in billions for each company
    max_employees = [50, 60, 70]  # Maximum employees each company can employ
    total_employees = 100  # Total employees available
    
    # CRITICAL: Validate array lengths to prevent IndexError
    assert len(profits) == len(max_employees), "Array length mismatch"
    safe_range = range(len(profits))  # Safe indexing
    
    # 2. VARIABLES
    # Decision variables for the number of employees allocated to each company
    x = {i: mdl.continuous_var(name=f"x_{i}", lb=0) for i in safe_range}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize total profits
    objective = mdl.sum(profits[i] * x[i] for i in safe_range)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS
    # Total employees constraint
    total_allocation = mdl.sum(x[i] for i in safe_range)
    mdl.add_constraint(total_allocation <= total_employees, ctname="total_employees")
    
    # Maximum employees per company constraints
    for i in safe_range:
        mdl.add_constraint(x[i] <= max_employees[i], ctname=f"max_employees_{i}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in safe_range:
            value = solution.get_value(x[i])
            if value > 1e-6:
                print(f"x[{i}] = {value:.3f}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
    
    return mdl

# Run the optimization
optimize_employee_allocation()