# Complete GUROBIPY implementation

import gurobipy as gp
from gurobipy import GRB

def optimize_employee_allocation():
    """Optimize employee allocation to maximize profits"""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("company_employee")
    
    # Data from the problem context
    profits = [1.5, 2.0, 3.0]  # Profits for Company 1, 2, and 3
    max_employees = [50, 60, 70]  # Max employees for Company 1, 2, and 3
    total_employees = 100  # Total available employees
    n_companies = len(profits)
    
    # CRITICAL: Validate array lengths before loops
    assert len(profits) == len(max_employees) == n_companies, "Array length mismatch"
    
    # 2. VARIABLES
    # Variable dictionaries for employee allocation
    x = {i: model.addVar(vtype=GRB.CONTINUOUS, name=f"x_{i+1}", lb=0) 
         for i in range(n_companies)}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize total profits
    model.setObjective(gp.quicksum(profits[i] * x[i] for i in range(n_companies)), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Total employees constraint
    model.addConstr(gp.quicksum(x[i] for i in range(n_companies)) <= total_employees, name="total_employees")
    
    # Max employees per company constraints
    for i in range(n_companies):
        model.addConstr(x[i] <= max_employees[i], name=f"max_employees_{i+1}")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in range(n_companies):
            if x[i].x > 1e-6:
                print(f"x[{i+1}] = {x[i].x:.3f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
optimize_employee_allocation()