# Complete PYOMO implementation

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def company_employee_optimization():
    """Optimize employee allocation to maximize profits"""
    
    # 1. MODEL CREATION
    model = pyo.ConcreteModel()
    
    # 2. DATA SETUP
    # Profits for each company
    profits = [1.5, 2.0, 3.0]
    # Maximum employees each company can employ
    max_employees = [50, 60, 70]
    # Total employees available
    total_employees = 100
    
    # CRITICAL: Validate array lengths before indexing
    n_companies = len(profits)
    assert len(max_employees) == n_companies, "Array length mismatch"
    
    # 3. SETS
    model.I = pyo.RangeSet(1, n_companies)  # 1-based indexing
    
    # 4. PARAMETERS
    model.profit = pyo.Param(model.I, initialize={i+1: profits[i] for i in range(n_companies)})
    model.max_employees = pyo.Param(model.I, initialize={i+1: max_employees[i] for i in range(n_companies)})
    
    # 5. VARIABLES
    model.x = pyo.Var(model.I, within=pyo.NonNegativeReals, bounds=(0, total_employees))
    
    # 6. OBJECTIVE FUNCTION
    def obj_rule(model):
        return sum(model.profit[i] * model.x[i] for i in model.I)
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.maximize)
    
    # 7. CONSTRAINTS
    
    # Total employees constraint
    def total_employees_rule(model):
        return sum(model.x[i] for i in model.I) <= total_employees
    model.total_employees_constraint = pyo.Constraint(rule=total_employees_rule)
    
    # Maximum employees per company constraint
    def max_employees_rule(model, i):
        return model.x[i] <= model.max_employees[i]
    model.max_employees_constraint = pyo.Constraint(model.I, rule=max_employees_rule)
    
    # 8. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')
    
    # Optional: Set solver options
    solver.options['TimeLimit'] = 300  # 5 minutes
    solver.options['MIPGap'] = 0.01    # 1% gap
    
    # Solve the model
    results = solver.solve(model, tee=True)  # tee=True shows solver output
    
    # 9. RESULT PROCESSING
    # Check solver status
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print("Optimal solution found!")
        print(f"Optimal value: {pyo.value(model.objective):.3f}")
        
        # Extract variable values
        print("\nVariable values:")
        for i in model.I:
            x_val = pyo.value(model.x[i])
            if x_val > 1e-6:  # Only print non-zero values
                print(f"x[{i}] = {x_val:.3f}")
        
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")
    
    return model

# Run the optimization
company_employee_optimization()