# Complete PYOMO implementation - Retry Attempt 4

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def optimize_police_allocation():
    # 1. MODEL CREATION
    model = pyo.ConcreteModel()

    # 2. DATA SETUP
    # Data from the county_public_safety table
    crime_rates = [0.15, 0.25, 0.18, 0.22, 0.2]
    populations = [1500, 2500, 1800, 2200, 2000]
    residents_per_officer = [150, 125, 180, 110, 100]
    case_burdens = [7.0, 12.0, 9.0, 11.0, 10.0]
    min_officers = [10, 20, 12, 18, 15]
    total_officers = 75
    max_case_burden = 10.0

    # Validate array lengths
    n_counties = len(crime_rates)
    assert len(populations) == len(residents_per_officer) == len(case_burdens) == len(min_officers) == n_counties, "Array length mismatch"

    # 3. SETS
    model.I = pyo.RangeSet(1, n_counties)

    # 4. PARAMETERS
    model.crime_rate = pyo.Param(model.I, initialize={i+1: crime_rates[i] for i in range(n_counties)})
    model.population = pyo.Param(model.I, initialize={i+1: populations[i] for i in range(n_counties)})
    model.residents_per_officer = pyo.Param(model.I, initialize={i+1: residents_per_officer[i] for i in range(n_counties)})
    model.case_burden = pyo.Param(model.I, initialize={i+1: case_burdens[i] for i in range(n_counties)})
    model.min_officers = pyo.Param(model.I, initialize={i+1: min_officers[i] for i in range(n_counties)})

    # 5. VARIABLES
    model.x = pyo.Var(model.I, within=pyo.NonNegativeIntegers)

    # 6. OBJECTIVE FUNCTION
    def obj_rule(model):
        return sum(model.crime_rate[i] * model.x[i] for i in model.I)
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.minimize)

    # 7. CONSTRAINTS
    def total_officers_rule(model):
        return sum(model.x[i] for i in model.I) <= total_officers
    model.total_officers_constraint = pyo.Constraint(rule=total_officers_rule)

    def min_officers_rule(model, i):
        return model.x[i] >= model.min_officers[i]
    model.min_officers_constraint = pyo.Constraint(model.I, rule=min_officers_rule)

    def residents_per_officer_rule(model, i):
        return model.x[i] <= model.population[i] / model.residents_per_officer[i]
    model.residents_per_officer_constraint = pyo.Constraint(model.I, rule=residents_per_officer_rule)

    def case_burden_rule(model, i):
        return model.x[i] >= model.case_burden[i] / max_case_burden
    model.case_burden_constraint = pyo.Constraint(model.I, rule=case_burden_rule)

    # 8. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')
    results = solver.solve(model, tee=True)

    # 9. RESULT PROCESSING
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print("Optimal solution found!")
        print(f"Optimal value: {pyo.value(model.objective)}")
        for i in model.I:
            print(f"x[{i}] = {pyo.value(model.x[i])}")
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")

    return model

# Execute the optimization
optimize_police_allocation()