# Complete GUROBIPY implementation

import gurobipy as gp
from gurobipy import GRB

def optimize_cultural_impact():
    """Optimize investment in book clubs and movies to maximize cultural impact."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("culture_company")
    
    # Coefficients for the objective function
    book_club_coeffs = [0.6, 0.8, 0.5]
    movie_coeffs = [0.7, 0.9, 0.4]
    
    # Budget constraint
    budget_constraint = 1000000
    
    # Validate array lengths
    n_book_clubs = len(book_club_coeffs)
    n_movies = len(movie_coeffs)
    assert n_book_clubs == 3, "Expected 3 book clubs"
    assert n_movies == 3, "Expected 3 movies"
    
    # 2. VARIABLES
    # Investment amounts in book clubs
    x = {i: model.addVar(vtype=GRB.CONTINUOUS, name=f"x_{i+1}", lb=0) 
         for i in range(n_book_clubs)}
    
    # Investment amounts in movies
    y = {i: model.addVar(vtype=GRB.CONTINUOUS, name=f"y_{i+1}", lb=0) 
         for i in range(n_movies)}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the total cultural impact
    model.setObjective(
        gp.quicksum(book_club_coeffs[i] * x[i] for i in range(n_book_clubs)) +
        gp.quicksum(movie_coeffs[i] * y[i] for i in range(n_movies)), 
        GRB.MAXIMIZE
    )
    
    # 4. CONSTRAINTS
    # Budget constraint
    model.addConstr(
        gp.quicksum(x[i] for i in range(n_book_clubs)) +
        gp.quicksum(y[i] for i in range(n_movies)) <= budget_constraint, 
        name="budget_constraint"
    )
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in range(n_book_clubs):
            if x[i].x > 1e-6:
                print(f"x[{i+1}] = {x[i].x:.3f}")
        for i in range(n_movies):
            if y[i].x > 1e-6:
                print(f"y[{i+1}] = {y[i].x:.3f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
optimize_cultural_impact()