# Complete GUROBIPY implementation

import gurobipy as gp
from gurobipy import GRB

def optimize_reviews():
    """Optimize the selection of user reviews to maximize trust score."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("epinions_1")
    
    # Data: Trust scores and ratings for each review
    trust_scores = [0.7, 0.9, 0.4, 0.3, 0.1]
    ratings = [5, 2, 4, 3, 1]
    max_reviews = 3
    n_reviews = len(trust_scores)
    
    # CRITICAL: Validate array lengths before loops
    assert len(trust_scores) == len(ratings) == n_reviews, "Array length mismatch"
    
    # 2. VARIABLES
    # Binary decision variables for each review
    x = {i: model.addVar(vtype=GRB.BINARY, name=f"x_{i}") for i in range(n_reviews)}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the overall trust score
    model.setObjective(gp.quicksum(trust_scores[i] * ratings[i] * x[i] for i in range(n_reviews)), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS
    # Constraint: Total number of selected reviews must not exceed max_reviews
    model.addConstr(gp.quicksum(x[i] for i in range(n_reviews)) <= max_reviews, name="max_reviews")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in range(n_reviews):
            if x[i].x > 1e-6:
                print(f"x[{i}] = {x[i].x:.3f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
optimize_reviews()