# Complete DOCPLEX implementation

from docplex.mp.model import Model

def optimize_flight_allocation():
    """Optimize flight allocation to maximize passenger transport"""

    # 1. MODEL & DATA SETUP
    mdl = Model(name="flight_4")
    
    # Data from the problem context
    passenger_capacities = [150, 200, 250]
    total_flights_available = 12
    max_flights_per_route = 5
    min_flights_per_route = 1
    
    # Validate array lengths
    n_routes = len(passenger_capacities)
    assert n_routes == 3, "Expected 3 routes based on problem data"
    safe_range = range(n_routes)  # Safe indexing
    
    # 2. VARIABLES
    # Integer variables for the number of flights assigned to each route
    flights = {i: mdl.integer_var(name=f"x_{i+1}", lb=min_flights_per_route, ub=max_flights_per_route) for i in safe_range}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the total number of passengers transported
    objective = mdl.sum(passenger_capacities[i] * flights[i] for i in safe_range)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS
    # Total number of flights assigned across all routes must not exceed the total available
    total_flights = mdl.sum(flights[i] for i in safe_range)
    mdl.add_constraint(total_flights <= total_flights_available, ctname="total_flights_limit")
    
    # Each route must have at least one flight assigned (already handled by variable bounds)
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in safe_range:
            value = solution.get_value(flights[i])
            print(f"Flights assigned to route {i+1}: {value}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")

    return mdl

# Run the optimization
optimize_flight_allocation()