# Complete GUROBIPY implementation

import gurobipy as gp
from gurobipy import GRB

def optimize_flight_allocation():
    """Optimize flight allocation to maximize passenger transport"""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("flight_4")
    
    # Data: Passenger capacities for each route
    passenger_capacities = [150, 200, 250]
    n_routes = len(passenger_capacities)
    
    # Validate array lengths
    assert n_routes == 3, "Expected 3 routes"
    
    # 2. VARIABLES
    # Decision variables: Number of flights assigned to each route
    flights = model.addVars(n_routes, vtype=GRB.INTEGER, name="flights", lb=1, ub=5)
    
    # 3. OBJECTIVE FUNCTION
    # Maximize total number of passengers transported
    model.setObjective(gp.quicksum(passenger_capacities[i] * flights[i] for i in range(n_routes)), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS
    # Total number of flights assigned must not exceed 12
    model.addConstr(gp.quicksum(flights[i] for i in range(n_routes)) <= 12, name="total_flights")
    
    # Each route must have at least one flight assigned (handled by lb=1 in variable definition)
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in range(n_routes):
            print(f"Route {i+1} flights: {flights[i].x:.0f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
optimize_flight_allocation()