# Complete Optimization Problem and Solution: flight_company

## 1. Problem Context and Goals

### Context  
The flight company is focused on optimizing the allocation of its flights to various airports to minimize the overall operational costs. The decision-making process involves determining whether a flight should be assigned to a specific airport, represented by a binary decision variable. The operational costs are influenced by the distance between the flight's origin and the airport, as well as the fee rate charged by the airport. The company aims to ensure that each flight is assigned to exactly one airport, and that no airport exceeds its capacity to handle flights. The cost of assigning a flight to an airport is calculated by multiplying the distance by the fee rate, which serves as the basis for the objective coefficients in the optimization model. This approach ensures that the problem remains linear, avoiding any nonlinear relationships such as variable products or divisions. The business configuration includes both scalar parameters and business logic formulas, ensuring consistency and flexibility in parameter tuning.

### Goals  
The primary goal of the optimization is to minimize the total operational cost associated with assigning flights to airports. This involves reducing the sum of costs calculated by multiplying the distance and fee rate for each flight-airport assignment. Success in this optimization is measured by achieving the lowest possible total cost while adhering to the constraints of flight assignments and airport capacities. The objective is clearly defined in linear terms, focusing on minimizing the total cost without involving any complex mathematical expressions.

## 2. Constraints    

The optimization problem is subject to several constraints that ensure the feasibility and practicality of the solution:

- Each flight must be assigned to exactly one airport. This ensures that all flights are accounted for and properly managed within the system.
- The total number of flights assigned to an airport must not exceed its capacity. This constraint ensures that airports operate within their limits, maintaining efficient and safe operations.

These constraints are expressed in business terms that naturally lead to linear mathematical forms, ensuring that the optimization problem remains straightforward and solvable using linear programming techniques.

## 3. Available Data  

### Database Schema  
```sql
-- Iteration 2 Database Schema
-- Objective: Schema changes include adding tables for distance and fee rate data, updating business configuration logic for cost coefficient formula, and ensuring all tables have at least 3 rows of data.

CREATE TABLE Flights (
  flight_id INTEGER,
  airport_id INTEGER,
  assignment BOOLEAN
);

CREATE TABLE Airports (
  airport_id INTEGER,
  capacity INTEGER
);

CREATE TABLE Distances (
  flight_id INTEGER,
  airport_id INTEGER,
  distance FLOAT
);

CREATE TABLE FeeRates (
  airport_id INTEGER,
  fee_rate FLOAT
);


```

### Data Dictionary  
The data dictionary provides a comprehensive overview of the tables and columns, explaining their business purposes and roles in the optimization process:

- **Flights Table**: Represents the assignment of flights to airports. It includes:
  - `flight_id`: A unique identifier for each flight, used to identify flights in the optimization model.
  - `airport_id`: An identifier for the assigned airport, linking flights to airports in the optimization model.
  - `assignment`: A boolean indicating if a flight is assigned to an airport, serving as a decision variable in the optimization model.

- **Airports Table**: Contains data about airports, including their capacity. It includes:
  - `airport_id`: A unique identifier for each airport, used to identify airports in the optimization model.
  - `capacity`: The maximum number of flights an airport can handle, serving as a constraint bound in the optimization model.

- **Distances Table**: Represents the distance between each flight and airport for cost calculation. It includes:
  - `flight_id`: A unique identifier for each flight, used in cost calculation.
  - `airport_id`: An identifier for the airport, used in cost calculation.
  - `distance`: The distance between the flight and airport, used in cost calculation.

- **FeeRates Table**: Represents the fee rate for each airport for cost calculation. It includes:
  - `airport_id`: An identifier for the airport, used in cost calculation.
  - `fee_rate`: The fee rate for using the airport, used in cost calculation.

### Current Stored Values  
```sql
-- Iteration 2 Realistic Data
-- Generated by triple expert (business + data + optimization)
-- Values were determined based on typical distances between major airports and realistic fee rates charged by airports. Capacities were set to reflect typical airport operations, ensuring a balance between demand and supply.

-- Realistic data for Flights
INSERT INTO Flights (flight_id, airport_id, assignment) VALUES (1, 101, True);
INSERT INTO Flights (flight_id, airport_id, assignment) VALUES (2, 102, True);
INSERT INTO Flights (flight_id, airport_id, assignment) VALUES (3, 103, True);

-- Realistic data for Airports
INSERT INTO Airports (airport_id, capacity) VALUES (101, 120);
INSERT INTO Airports (airport_id, capacity) VALUES (102, 180);
INSERT INTO Airports (airport_id, capacity) VALUES (103, 150);

-- Realistic data for Distances
INSERT INTO Distances (flight_id, airport_id, distance) VALUES (1, 101, 550.0);
INSERT INTO Distances (flight_id, airport_id, distance) VALUES (2, 102, 800.0);
INSERT INTO Distances (flight_id, airport_id, distance) VALUES (3, 103, 950.0);

-- Realistic data for FeeRates
INSERT INTO FeeRates (airport_id, fee_rate) VALUES (101, 12.0);
INSERT INTO FeeRates (airport_id, fee_rate) VALUES (102, 18.0);
INSERT INTO FeeRates (airport_id, fee_rate) VALUES (103, 22.0);


```

## 4. Mathematical Optimization Formulation

#### Decision Variables
- \( x_{ij} \): Binary decision variable indicating whether flight \( i \) is assigned to airport \( j \). \( x_{ij} = 1 \) if flight \( i \) is assigned to airport \( j \), otherwise \( x_{ij} = 0 \).

#### Objective Function
Minimize the total operational cost of assigning flights to airports:

\[
\text{Minimize } \sum_{i,j} \text{distance}_{ij} \times \text{fee\_rate}_j \times x_{ij}
\]

Where:
- \(\text{distance}_{ij}\) is the distance between flight \( i \) and airport \( j \) from the `Distances` table.
- \(\text{fee\_rate}_j\) is the fee rate for airport \( j \) from the `FeeRates` table.

#### Constraints
1. **Each flight must be assigned to exactly one airport:**

\[
\sum_{j} x_{ij} = 1 \quad \forall i
\]

2. **The total number of flights assigned to an airport must not exceed its capacity:**

\[
\sum_{i} x_{ij} \leq \text{capacity}_j \quad \forall j
\]

Where:
- \(\text{capacity}_j\) is the capacity of airport \( j \) from the `Airports` table.

#### Data Source Verification:
- \(\text{distance}_{ij}\) is sourced from `Distances.distance`.
- \(\text{fee\_rate}_j\) is sourced from `FeeRates.fee_rate`.
- \(\text{capacity}_j\) is sourced from `Airports.capacity`.

This formulation provides a complete linear programming model that can be solved using standard optimization software to determine the optimal assignment of flights to airports, minimizing the total operational cost while respecting all constraints.

## 5. Gurobipy Implementation

```python
# Complete GUROBIPY implementation - Retry Attempt 4

import gurobipy as gp
from gurobipy import GRB

def flight_company_optimization():
    """Optimize flight assignments to minimize operational costs."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("flight_company")
    
    # Example data setup (replace with actual data retrieval logic)
    flight_ids = [1, 2, 3]
    airport_ids = [101, 102, 103]
    
    distances = {
        (1, 101): 550.0,
        (2, 102): 800.0,
        (3, 103): 950.0
    }
    
    fee_rates = {
        101: 12.0,
        102: 18.0,
        103: 22.0
    }
    
    capacities = {
        101: 120,
        102: 180,
        103: 150
    }
    
    # CRITICAL: Validate data lengths
    assert len(flight_ids) * len(airport_ids) == len(distances), "Data length mismatch"
    
    # 2. VARIABLES
    x = model.addVars(flight_ids, airport_ids, vtype=GRB.BINARY, name="x")
    
    # 3. OBJECTIVE FUNCTION
    model.setObjective(
        gp.quicksum(distances[i, j] * fee_rates[j] * x[i, j] for i in flight_ids for j in airport_ids),
        GRB.MINIMIZE
    )
    
    # 4. CONSTRAINTS
    # Each flight must be assigned to exactly one airport
    model.addConstrs(
        (gp.quicksum(x[i, j] for j in airport_ids) == 1 for i in flight_ids),
        name="assign_flight"
    )
    
    # The total number of flights assigned to an airport must not exceed its capacity
    model.addConstrs(
        (gp.quicksum(x[i, j] for i in flight_ids) <= capacities[j] for j in airport_ids),
        name="capacity_limit"
    )
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in flight_ids:
            for j in airport_ids:
                if x[i, j].x > 1e-6:
                    print(f"Flight {i} assigned to Airport {j}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    else:
        print("Optimization ended with status", model.status)

# Run the optimization
flight_company_optimization()
```

### Execution Results
**Status**: ERROR
**Error**: Traceback (most recent call last):
  File "/tmp/tmpx906n5r7.py", line 76, in <module>
    flight_company_optimization()
  File "/tmp/tmpx906n5r7.py", line 35, in flight_company_optimization
    assert len(flight_ids) * len(airport_ids) == len(distances), "Data length mismatch"
AssertionError: Data length mismatch

**Analysis**: Gurobipy encountered a data length mismatch error, indicating a potential issue with the input data structure or data integrity. The solver could not proceed to find a solution.

## 6. DOCplex Implementation

```python
# Complete DOCPLEX implementation - Retry Attempt 4

from docplex.mp.model import Model

def flight_company_optimization():
    # 1. MODEL & DATA SETUP
    mdl = Model(name="flight_company")
    
    # Data from the problem statement
    flight_ids = [1, 2, 3]
    airport_ids = [101, 102, 103]
    distances = {
        (1, 101): 550.0,
        (2, 102): 800.0,
        (3, 103): 950.0
    }
    fee_rates = {
        101: 12.0,
        102: 18.0,
        103: 22.0
    }
    capacities = {
        101: 120,
        102: 180,
        103: 150
    }
    
    # CRITICAL: Validate data consistency
    assert len(flight_ids) * len(airport_ids) == len(distances), "Data length mismatch"
    
    # 2. VARIABLES
    x = {(i, j): mdl.binary_var(name=f"x_{i}_{j}") for i in flight_ids for j in airport_ids}
    
    # 3. OBJECTIVE FUNCTION
    objective = mdl.sum(distances[i, j] * fee_rates[j] * x[i, j] for i in flight_ids for j in airport_ids)
    mdl.minimize(objective)
    
    # 4. CONSTRAINTS
    # Each flight must be assigned to exactly one airport
    for i in flight_ids:
        mdl.add_constraint(mdl.sum(x[i, j] for j in airport_ids) == 1, ctname=f"assign_flight_{i}")
    
    # The total number of flights assigned to an airport must not exceed its capacity
    for j in airport_ids:
        mdl.add_constraint(mdl.sum(x[i, j] for i in flight_ids) <= capacities[j], ctname=f"capacity_airport_{j}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in flight_ids:
            for j in airport_ids:
                if solution.get_value(x[i, j]) > 0.5:  # Binary variable, check if assigned
                    print(f"Flight {i} assigned to Airport {j}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")

# Run the optimization
flight_company_optimization()
```

### Execution Results
**Status**: ERROR
**Error**: Traceback (most recent call last):
  File "/tmp/tmpth908u56.py", line 61, in <module>
    flight_company_optimization()
  File "/tmp/tmpth908u56.py", line 29, in flight_company_optimization
    assert len(flight_ids) * len(airport_ids) == len(distances), "Data length mismatch"
AssertionError: Data length mismatch

**Analysis**: DOCplex also faced a data length mismatch error, similar to Gurobipy, suggesting a consistent issue across solvers related to the input data.

## 7. Pyomo Implementation

```python
# Complete PYOMO implementation - Retry Attempt 4

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def flight_company_optimization():
    """Optimize flight assignments to minimize operational costs."""
    
    # 1. MODEL CREATION
    model = pyo.ConcreteModel()
    
    # 2. DATA SETUP
    # Define the data based on the provided database schema and current stored values
    flight_ids = [1, 2, 3]
    airport_ids = [101, 102, 103]
    
    distances = {
        (1, 101): 550.0,
        (2, 102): 800.0,
        (3, 103): 950.0
    }
    
    fee_rates = {
        101: 12.0,
        102: 18.0,
        103: 22.0
    }
    
    capacities = {
        101: 120,
        102: 180,
        103: 150
    }
    
    # CRITICAL: Validate data consistency
    assert all((i, j) in distances for i in flight_ids for j in airport_ids), "Data length mismatch"
    
    # 3. SETS
    model.F = pyo.Set(initialize=flight_ids)
    model.A = pyo.Set(initialize=airport_ids)
    
    # 4. PARAMETERS
    model.distance = pyo.Param(model.F * model.A, initialize=distances, within=pyo.NonNegativeReals)
    model.fee_rate = pyo.Param(model.A, initialize=fee_rates, within=pyo.NonNegativeReals)
    model.capacity = pyo.Param(model.A, initialize=capacities, within=pyo.NonNegativeIntegers)
    
    # 5. VARIABLES
    model.x = pyo.Var(model.F, model.A, within=pyo.Binary)
    
    # 6. OBJECTIVE FUNCTION
    def obj_rule(model):
        return sum(model.distance[i, j] * model.fee_rate[j] * model.x[i, j] for i in model.F for j in model.A)
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.minimize)
    
    # 7. CONSTRAINTS
    # Each flight must be assigned to exactly one airport
    def flight_assignment_rule(model, i):
        return sum(model.x[i, j] for j in model.A) == 1
    model.flight_assignment_constraint = pyo.Constraint(model.F, rule=flight_assignment_rule)
    
    # The total number of flights assigned to an airport must not exceed its capacity
    def airport_capacity_rule(model, j):
        return sum(model.x[i, j] for i in model.F) <= model.capacity[j]
    model.airport_capacity_constraint = pyo.Constraint(model.A, rule=airport_capacity_rule)
    
    # 8. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')
    
    # Solve the model
    results = solver.solve(model, tee=True)
    
    # 9. RESULT PROCESSING
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print("Optimal solution found!")
        print(f"Optimal value: {pyo.value(model.objective)}")
        
        # Extract variable values
        print("\nFlight assignments:")
        for i in model.F:
            for j in model.A:
                if pyo.value(model.x[i, j]) > 0.5:  # Binary variable, check if assigned
                    print(f"Flight {i} assigned to Airport {j}")
        
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")

# Run the optimization
flight_company_optimization()
```

### Execution Results
**Status**: ERROR
**Error**: Traceback (most recent call last):
  File "/tmp/tmp_wb0m3_x.py", line 92, in <module>
    flight_company_optimization()
  File "/tmp/tmp_wb0m3_x.py", line 36, in flight_company_optimization
    assert all((i, j) in distances for i in flight_ids for j in airport_ids), "Data length mismatch"
AssertionError: Data length mismatch

**Analysis**: Pyomo reported a data length mismatch error, indicating that the input data does not align with the expected format or content, preventing the solver from executing successfully.

## 8. Cross-Solver Analysis and Final Recommendation

### Solver Results Comparison

| Solver | Status | Optimal Value | Execution Time | Decision Variables | Retry Attempt |
|--------|--------|---------------|----------------|-------------------|---------------|
| Gurobipy | ERROR | N/A | 0.45s | N/A | 4 |
| Docplex | ERROR | N/A | 3.84s | N/A | 4 |
| Pyomo | ERROR | N/A | 2.46s | N/A | 4 |

### Solver Consistency Analysis
**Result**: Solvers produced inconsistent results
**Inconsistent Solvers**: gurobipy, docplex, pyomo
**Potential Issues**:
- Data length mismatch suggests that the input data tables (Distances, FeeRates, Airports) may not be correctly populated or aligned.
- There may be missing or extra entries in the Distances table compared to the expected number of flight-airport pairs.
**Solver Retry Summary**: gurobipy: 4 attempts, docplex: 4 attempts, pyomo: 4 attempts

### Final Recommendation
**Confidence Level**: LOW
**Preferred Solver(s)**: multiple
**Reasoning**: All solvers encountered the same data issue, indicating that the problem lies with the input data rather than the solvers themselves. Therefore, no solver preference can be made until the data issue is resolved.

### Business Interpretation
**Overall Strategy**: The current optimization attempt failed due to data issues, and no valid assignment of flights to airports could be determined.
**Objective Value Meaning**: The optimal objective value could not be calculated due to errors in the input data, preventing any meaningful business interpretation.
**Resource Allocation Summary**: Resource allocation could not be determined due to the failure in solving the optimization problem.
**Implementation Recommendations**: Verify and correct the input data tables to ensure that all expected flight-airport pairs are present and correctly formatted. Once data integrity is ensured, re-run the optimization using the solvers.