# Complete Optimization Problem and Solution: game_1

## 1. Problem Context and Goals

### Context  
The university is focused on optimizing how students allocate their time between playing video games and participating in sports. The aim is to enhance student engagement while ensuring that academic responsibilities are not neglected. The decision variables in this context are the hours students allocate to games and sports, represented as continuous variables. Specifically, x[StuID, GameID] denotes the hours a student allocates to a particular game, and y[StuID, SportName] represents the hours allocated to a specific sport. The operational parameters are designed to align with the objective of maximizing total engagement. This involves summing the product of game engagement scores and hours allocated to games, along with the product of sport engagement scores and hours allocated to sports. The business configuration includes constraints such as the total hours a student can allocate to games and sports, the minimum hours a student on scholarship must spend on sports, and the maximum hours a student can spend on games. These constraints ensure that the allocation of time is balanced and adheres to the university's academic and extracurricular policies.

### Goals  
The primary goal of this optimization problem is to maximize student engagement. This is achieved by optimizing the allocation of hours to games and sports in a way that maximizes the total engagement score. The engagement score is calculated by summing the products of game engagement scores with the hours allocated to games and sport engagement scores with the hours allocated to sports. Success in this optimization is measured by the total engagement score, which reflects the effectiveness of the time allocation in enhancing student engagement. The objective is clearly defined in linear terms, focusing on maximizing this engagement score without involving any nonlinear relationships.

## 2. Constraints    

The constraints in this optimization problem are designed to ensure that the allocation of time adheres to the university's policies and supports student success. The first constraint ensures that the total hours a student allocates to games and sports do not exceed the available hours, reflecting a balanced schedule. The second constraint mandates that students on scholarships spend a minimum number of hours on sports, ensuring they meet participation requirements. The third constraint limits the hours a student can spend on games, preventing excessive gaming and promoting academic focus. These constraints are expressed in linear terms, directly aligning with the expected mathematical formulation.

## 3. Available Data  

### Database Schema  
```sql
-- Iteration 2 Database Schema
-- Objective: Schema changes include creating new tables for decision variables, modifying existing tables to improve mapping adequacy, and moving scalar parameters to configuration logic.

CREATE TABLE EngagementScores (
  StuID INTEGER,
  GameID INTEGER,
  SportName STRING,
  GameEngagementScore FLOAT,
  SportEngagementScore FLOAT
);

CREATE TABLE GameAllocation (
  StuID INTEGER,
  GameID INTEGER,
  HoursAllocated FLOAT
);

CREATE TABLE SportAllocation (
  StuID INTEGER,
  SportName STRING,
  HoursAllocated FLOAT
);
```

### Data Dictionary  
The data dictionary provides a comprehensive overview of the tables and columns used in this optimization problem, mapping them to their business purposes and roles in the optimization process. 

- **EngagementScores**: This table stores the engagement scores for each student, game, and sport. The engagement scores serve as coefficients in the objective function, linking the scores to the students, games, and sports.
  - **StuID**: Represents the student identifier, linking engagement scores to individual students.
  - **GameID**: Represents the game identifier, linking engagement scores to specific games.
  - **SportName**: Represents the name of the sport, linking engagement scores to specific sports.
  - **GameEngagementScore**: Represents the engagement score for games, used as a coefficient in the objective function.
  - **SportEngagementScore**: Represents the engagement score for sports, used as a coefficient in the objective function.

- **GameAllocation**: This table records the hours students allocate to games, serving as decision variables in the optimization model.
  - **StuID**: Represents the student identifier, linking allocation to individual students.
  - **GameID**: Represents the game identifier, linking allocation to specific games.
  - **HoursAllocated**: Represents the hours allocated by a student to a game, serving as a decision variable.

- **SportAllocation**: This table records the hours students allocate to sports, serving as decision variables in the optimization model.
  - **StuID**: Represents the student identifier, linking allocation to individual students.
  - **SportName**: Represents the name of the sport, linking allocation to specific sports.
  - **HoursAllocated**: Represents the hours allocated by a student to a sport, serving as a decision variable.

### Current Stored Values  
```sql
-- Iteration 2 Realistic Data
-- Generated by triple expert (business + data + optimization)
-- Values were determined based on typical student schedules, engagement levels, and scholarship requirements to ensure a balanced allocation between games and sports.

-- Realistic data for EngagementScores
INSERT INTO EngagementScores (StuID, GameID, SportName, GameEngagementScore, SportEngagementScore) VALUES (1, 101, 'Basketball', 1.8, 1.5);
INSERT INTO EngagementScores (StuID, GameID, SportName, GameEngagementScore, SportEngagementScore) VALUES (2, 102, 'Soccer', 2.2, 1.8);
INSERT INTO EngagementScores (StuID, GameID, SportName, GameEngagementScore, SportEngagementScore) VALUES (3, 103, 'Tennis', 1.5, 2.0);

-- Realistic data for GameAllocation
INSERT INTO GameAllocation (StuID, GameID, HoursAllocated) VALUES (1, 101, 10.0);
INSERT INTO GameAllocation (StuID, GameID, HoursAllocated) VALUES (2, 102, 12.0);
INSERT INTO GameAllocation (StuID, GameID, HoursAllocated) VALUES (3, 103, 8.0);

-- Realistic data for SportAllocation
INSERT INTO SportAllocation (StuID, SportName, HoursAllocated) VALUES (1, 'Basketball', 8.0);
INSERT INTO SportAllocation (StuID, SportName, HoursAllocated) VALUES (2, 'Soccer', 6.0);
INSERT INTO SportAllocation (StuID, SportName, HoursAllocated) VALUES (3, 'Tennis', 10.0);
```

## 4. Mathematical Optimization Formulation

#### Decision Variables
- \( x[StuID, GameID] \): Continuous variable representing the hours a student (StuID) allocates to a particular game (GameID).
- \( y[StuID, SportName] \): Continuous variable representing the hours a student (StuID) allocates to a specific sport (SportName).

#### Objective Function
Maximize the total engagement score:
\[
\text{Maximize } \sum_{StuID, GameID} \text{GameEngagementScore}[StuID, GameID] \times x[StuID, GameID] + \sum_{StuID, SportName} \text{SportEngagementScore}[StuID, SportName] \times y[StuID, SportName]
\]

Data Source Verification:
- \(\text{GameEngagementScore}[StuID, GameID]\) from `EngagementScores.GameEngagementScore`
- \(\text{SportEngagementScore}[StuID, SportName]\) from `EngagementScores.SportEngagementScore`

#### Constraints
1. **Total Hours Constraint**: The total hours a student allocates to games and sports should not exceed the available hours (assumed to be 20 hours per week for this example).
   \[
   \sum_{GameID} x[StuID, GameID] + \sum_{SportName} y[StuID, SportName] \leq 20, \quad \forall \, StuID
   \]

2. **Scholarship Sports Hours Constraint**: Students on scholarships must spend a minimum number of hours on sports (assumed to be 5 hours per week for this example).
   \[
   \sum_{SportName} y[StuID, SportName] \geq 5, \quad \forall \, StuID \text{ on scholarship}
   \]

3. **Maximum Gaming Hours Constraint**: The hours a student can spend on games should not exceed a maximum limit (assumed to be 15 hours per week for this example).
   \[
   \sum_{GameID} x[StuID, GameID] \leq 15, \quad \forall \, StuID
   \]

Data Source Verification:
- Total hours, scholarship hours, and maximum gaming hours are assumed parameters based on typical university policies and are not directly sourced from the provided tables.

This linear model is immediately solvable using linear programming techniques, ensuring that the allocation of time maximizes student engagement while adhering to university policies.

## 5. Gurobipy Implementation

```python
# Complete GUROBIPY implementation

import gurobipy as gp
from gurobipy import GRB

def optimize_student_engagement():
    """Optimize student engagement by allocating hours to games and sports."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("student_engagement")
    
    # Sample data based on the provided problem context
    students = [1, 2, 3]
    games = [101, 102, 103]
    sports = ['Basketball', 'Soccer', 'Tennis']
    
    # Engagement scores from the database
    game_engagement_scores = {
        (1, 101): 1.8,
        (2, 102): 2.2,
        (3, 103): 1.5
    }
    
    sport_engagement_scores = {
        (1, 'Basketball'): 1.5,
        (2, 'Soccer'): 1.8,
        (3, 'Tennis'): 2.0
    }
    
    # 2. VARIABLES
    # Decision variables for hours allocated to games and sports
    x = model.addVars(students, games, vtype=GRB.CONTINUOUS, name="x", lb=0)
    y = model.addVars(students, sports, vtype=GRB.CONTINUOUS, name="y", lb=0)
    
    # 3. OBJECTIVE FUNCTION
    # Maximize total engagement score
    model.setObjective(
        gp.quicksum(game_engagement_scores[StuID, GameID] * x[StuID, GameID] for StuID in students for GameID in games) +
        gp.quicksum(sport_engagement_scores[StuID, SportName] * y[StuID, SportName] for StuID in students for SportName in sports),
        GRB.MAXIMIZE
    )
    
    # 4. CONSTRAINTS
    # Total hours constraint
    model.addConstrs(
        (gp.quicksum(x[StuID, GameID] for GameID in games) + gp.quicksum(y[StuID, SportName] for SportName in sports) <= 20
         for StuID in students),
        name="total_hours"
    )
    
    # Scholarship sports hours constraint (assuming all students are on scholarship for simplicity)
    model.addConstrs(
        (gp.quicksum(y[StuID, SportName] for SportName in sports) >= 5 for StuID in students),
        name="scholarship_sports_hours"
    )
    
    # Maximum gaming hours constraint
    model.addConstrs(
        (gp.quicksum(x[StuID, GameID] for GameID in games) <= 15 for StuID in students),
        name="max_gaming_hours"
    )
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for StuID in students:
            for GameID in games:
                if x[StuID, GameID].x > 1e-6:
                    print(f"x[{StuID}, {GameID}] = {x[StuID, GameID].x:.3f}")
            for SportName in sports:
                if y[StuID, SportName].x > 1e-6:
                    print(f"y[{StuID}, {SportName}] = {y[StuID, SportName].x:.3f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
optimize_student_engagement()
```

### Execution Results
**Status**: ERROR
**Error**: Traceback (most recent call last):
  File "/tmp/tmpw6dojmm4.py", line 83, in <module>
    optimize_student_engagement()
  File "/tmp/tmpw6dojmm4.py", line 38, in optimize_student_engagement
    gp.quicksum(game_engagement_scores[StuID, GameID] * x[StuID, GameID] for StuID in students for GameID in games) +
  File "src/gurobipy/_helpers.pyx", line 41, in gurobipy._helpers.quicksum
  File "/tmp/tmpw6dojmm4.py", line 38, in <genexpr>
    gp.quicksum(game_engagement_scores[StuID, GameID] * x[StuID, GameID] for StuID in students for GameID in games) +
KeyError: (1, 102)

**Analysis**: Gurobipy encountered a KeyError, indicating a possible mismatch between the indices used in the model and the data provided. This error suggests that the data structure or the indices for accessing the engagement scores might be incorrect or incomplete.

## 6. DOCplex Implementation

```python
# Complete DOCPLEX implementation

from docplex.mp.model import Model

def optimize_student_engagement():
    """Optimize student engagement by allocating hours to games and sports."""
    
    # 1. MODEL & DATA SETUP
    mdl = Model(name="student_engagement")
    
    # Sample data based on the provided problem context
    students = [1, 2, 3]
    games = [101, 102, 103]
    sports = ['Basketball', 'Soccer', 'Tennis']
    
    # Engagement scores (from the database schema)
    game_engagement_scores = {
        (1, 101): 1.8,
        (2, 102): 2.2,
        (3, 103): 1.5
    }
    
    sport_engagement_scores = {
        (1, 'Basketball'): 1.5,
        (2, 'Soccer'): 1.8,
        (3, 'Tennis'): 2.0
    }
    
    # CRITICAL: Validate array lengths to prevent IndexError
    assert len(game_engagement_scores) == len(sport_engagement_scores) == len(students), "Array length mismatch"
    safe_range_students = range(len(students))
    
    # 2. VARIABLES
    x = {(stu, game): mdl.continuous_var(name=f"x_{stu}_{game}", lb=0) for stu in students for game in games}
    y = {(stu, sport): mdl.continuous_var(name=f"y_{stu}_{sport}", lb=0) for stu in students for sport in sports}
    
    # 3. OBJECTIVE FUNCTION
    objective = mdl.sum(game_engagement_scores[(stu, game)] * x[(stu, game)] for stu, game in game_engagement_scores) + \
                mdl.sum(sport_engagement_scores[(stu, sport)] * y[(stu, sport)] for stu, sport in sport_engagement_scores)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS
    # Total Hours Constraint
    for stu in students:
        mdl.add_constraint(
            mdl.sum(x[(stu, game)] for game in games) + mdl.sum(y[(stu, sport)] for sport in sports) <= 20,
            ctname=f"total_hours_{stu}"
        )
    
    # Scholarship Sports Hours Constraint (assuming all students are on scholarship for simplicity)
    for stu in students:
        mdl.add_constraint(
            mdl.sum(y[(stu, sport)] for sport in sports) >= 5,
            ctname=f"scholarship_hours_{stu}"
        )
    
    # Maximum Gaming Hours Constraint
    for stu in students:
        mdl.add_constraint(
            mdl.sum(x[(stu, game)] for game in games) <= 15,
            ctname=f"max_gaming_hours_{stu}"
        )
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for stu in students:
            for game in games:
                value = solution.get_value(x[(stu, game)])
                if value > 1e-6:
                    print(f"x[{stu}, {game}] = {value:.3f}")
            for sport in sports:
                value = solution.get_value(y[(stu, sport)])
                if value > 1e-6:
                    print(f"y[{stu}, {sport}] = {value:.3f}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")

    return mdl

# Run the optimization
optimize_student_engagement()
```

### Execution Results
**Status**: OPTIMAL
**Optimal Value**: 116.5
**Execution Time**: 1.06 seconds
**Reliability**: high
**Analysis**: DOCplex successfully solved the problem and found an optimal solution with an objective value of 116.5. This indicates that the model and data were correctly interpreted by DOCplex.

## 7. Pyomo Implementation

```python
# Complete PYOMO implementation

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def student_engagement_optimization():
    """Optimize student engagement by allocating hours to games and sports."""
    
    # 1. MODEL CREATION
    model = pyo.ConcreteModel()
    
    # 2. DATA SETUP
    # Example data based on the problem context
    students = [1, 2, 3]
    games = [101, 102, 103]
    sports = ['Basketball', 'Soccer', 'Tennis']
    
    game_engagement_scores = {
        (1, 101): 1.8,
        (2, 102): 2.2,
        (3, 103): 1.5
    }
    
    sport_engagement_scores = {
        (1, 'Basketball'): 1.5,
        (2, 'Soccer'): 1.8,
        (3, 'Tennis'): 2.0
    }
    
    # 3. SETS
    model.Students = pyo.Set(initialize=students)
    model.Games = pyo.Set(initialize=games)
    model.Sports = pyo.Set(initialize=sports)
    
    # 4. PARAMETERS
    model.GameEngagementScore = pyo.Param(model.Students, model.Games, initialize=game_engagement_scores, default=0)
    model.SportEngagementScore = pyo.Param(model.Students, model.Sports, initialize=sport_engagement_scores, default=0)
    
    # 5. VARIABLES
    model.x = pyo.Var(model.Students, model.Games, within=pyo.NonNegativeReals)
    model.y = pyo.Var(model.Students, model.Sports, within=pyo.NonNegativeReals)
    
    # 6. OBJECTIVE FUNCTION
    def obj_rule(model):
        return sum(model.GameEngagementScore[s, g] * model.x[s, g] for s in model.Students for g in model.Games) + \
               sum(model.SportEngagementScore[s, sp] * model.y[s, sp] for s in model.Students for sp in model.Sports)
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.maximize)
    
    # 7. CONSTRAINTS
    
    # Total Hours Constraint
    def total_hours_rule(model, s):
        return sum(model.x[s, g] for g in model.Games) + sum(model.y[s, sp] for sp in model.Sports) <= 20
    model.total_hours_constraint = pyo.Constraint(model.Students, rule=total_hours_rule)
    
    # Scholarship Sports Hours Constraint
    def scholarship_sports_rule(model, s):
        # Assuming all students are on scholarship for this example
        return sum(model.y[s, sp] for sp in model.Sports) >= 5
    model.scholarship_sports_constraint = pyo.Constraint(model.Students, rule=scholarship_sports_rule)
    
    # Maximum Gaming Hours Constraint
    def max_gaming_hours_rule(model, s):
        return sum(model.x[s, g] for g in model.Games) <= 15
    model.max_gaming_hours_constraint = pyo.Constraint(model.Students, rule=max_gaming_hours_rule)
    
    # 8. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')
    
    # Solve the model
    results = solver.solve(model, tee=True)
    
    # 9. RESULT PROCESSING
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print("Optimal solution found!")
        print(f"Optimal value: {pyo.value(model.objective):.3f}")
        
        # Extract variable values
        print("\nVariable values:")
        for s in model.Students:
            for g in model.Games:
                x_val = pyo.value(model.x[s, g])
                if x_val > 1e-6:
                    print(f"x[{s}, {g}] = {x_val:.3f}")
            for sp in model.Sports:
                y_val = pyo.value(model.y[s, sp])
                if y_val > 1e-6:
                    print(f"y[{s}, {sp}] = {y_val:.3f}")
        
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")
    
    return model

# Run the optimization
student_engagement_optimization()
```

### Execution Results
**Status**: OPTIMAL
**Optimal Value**: 116.5
**Execution Time**: 1.02 seconds
**Reliability**: high
**Analysis**: Pyomo also successfully solved the problem with the same optimal value as DOCplex, suggesting consistency in the model's formulation and data interpretation.

## 8. Cross-Solver Analysis and Final Recommendation

### Solver Results Comparison

| Solver | Status | Optimal Value | Execution Time | Decision Variables | Retry Attempt |
|--------|--------|---------------|----------------|-------------------|---------------|
| Gurobipy | ERROR | N/A | 0.19s | N/A | N/A |
| Docplex | OPTIMAL | 116.50 | 1.06s | N/A | N/A |
| Pyomo | OPTIMAL | 116.50 | 1.02s | N/A | N/A |

### Solver Consistency Analysis
**Result**: All solvers produced consistent results ✓
**Consistent Solvers**: docplex, pyomo
**Majority Vote Optimal Value**: 116.5

### Final Recommendation
**Recommended Optimal Value**: 116.5
**Confidence Level**: HIGH
**Preferred Solver(s)**: multiple
**Reasoning**: Both DOCplex and Pyomo provided consistent and optimal results, indicating reliability in their solutions. Gurobipy's error suggests a need for data verification before use.

### Business Interpretation
**Overall Strategy**: The optimal engagement score of 116.5 indicates the maximum achievable student engagement under the given constraints. This reflects an efficient allocation of student time to games and sports.
**Objective Value Meaning**: The optimal objective value of 116.5 represents the highest possible engagement score achievable by optimally distributing student hours across games and sports.
**Resource Allocation Summary**: Students should allocate their time to maximize engagement scores while adhering to constraints, such as minimum sports hours for scholarship students and maximum gaming hours.
**Implementation Recommendations**: Ensure data integrity and correct indexing before running optimization. Use DOCplex or Pyomo for reliable results. Regularly update engagement scores to reflect current student interests.