# Complete GUROBIPY implementation

import gurobipy as gp
from gurobipy import GRB

def optimize_gas_company_allocation():
    """Optimize gas station allocation to maximize market value"""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("gas_company")
    
    # Data from the problem context
    company_ids = [1, 2, 3]
    market_values = [12.5, 18.0, 14.0]
    
    # Assume sales, profits, and assets data for each company
    sales = [40.0, 50.0, 30.0]  # Example sales data
    profits = [20.0, 25.0, 15.0]  # Example profits data
    assets = [80.0, 90.0, 70.0]  # Example assets data
    
    # Constraint bounds from the database
    total_sales_limit = 120.0
    minimum_profits_requirement = 60.0
    total_assets_limit = 250.0
    
    # CRITICAL: Validate array lengths before loops
    assert len(company_ids) == len(market_values) == len(sales) == len(profits) == len(assets), "Array length mismatch"
    
    # 2. VARIABLES
    x = {i: model.addVar(vtype=GRB.BINARY, name=f"x_{i}") for i in company_ids}
    
    # 3. OBJECTIVE FUNCTION
    model.setObjective(gp.quicksum(market_values[i-1] * x[i] for i in company_ids), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Total Sales Limit Constraint
    model.addConstr(gp.quicksum(sales[i-1] * x[i] for i in company_ids) <= total_sales_limit, name="total_sales_limit")
    
    # Minimum Profits Requirement Constraint
    model.addConstr(gp.quicksum(profits[i-1] * x[i] for i in company_ids) >= minimum_profits_requirement, name="minimum_profits_requirement")
    
    # Total Assets Limit Constraint
    model.addConstr(gp.quicksum(assets[i-1] * x[i] for i in company_ids) <= total_assets_limit, name="total_assets_limit")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in company_ids:
            if x[i].x > 1e-6:
                print(f"x[{i}] = {x[i].x:.3f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
optimize_gas_company_allocation()