# Complete DOCPLEX implementation - Retry Attempt 4

from docplex.mp.model import Model

def optimize_inn_revenue():
    # 1. MODEL & DATA SETUP
    mdl = Model(name="inn_revenue_optimization")
    
    # Data from the problem statement
    reservation_ids = [1, 2, 3]
    rates = [120.0, 150.0, 200.0]
    lengths_of_stay = [2, 3, 5]
    minimum_rate = 100.0  # Example minimum rate
    room_capacity = 4     # Example room capacity (not used directly in this model)
    
    # CRITICAL: Validate array lengths to prevent IndexError
    assert len(reservation_ids) == len(rates) == len(lengths_of_stay), "Array length mismatch"
    safe_range = range(len(reservation_ids))  # Safe indexing
    
    # 2. VARIABLES
    # Decision variables for rates
    r = {i: mdl.continuous_var(name=f"rate_{reservation_ids[i]}", lb=minimum_rate) for i in safe_range}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize total revenue
    total_revenue = mdl.sum(r[i] * lengths_of_stay[i] for i in safe_range)
    mdl.maximize(total_revenue)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Minimum rate constraint
    for i in safe_range:
        mdl.add_constraint(r[i] >= minimum_rate, ctname=f"min_rate_{reservation_ids[i]}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in safe_range:
            value = solution.get_value(r[i])
            print(f"Rate for reservation {reservation_ids[i]}: {value:.2f}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")

# Run the optimization
optimize_inn_revenue()