# Complete GUROBIPY implementation - Retry Attempt 4

import gurobipy as gp
from gurobipy import GRB

def optimize_inn_revenue():
    # 1. MODEL & DATA SETUP
    model = gp.Model("inn_1")
    
    # Data from the problem context
    reservation_ids = [1, 2, 3]
    rates = [120.0, 150.0, 200.0]
    lengths_of_stay = [2, 3, 5]
    minimum_rate = 100.0  # Example minimum rate
    room_capacity = 4  # Example room capacity
    occupancies = [2, 3, 4]  # Example occupancies for each reservation
    
    # Validate array lengths
    assert len(reservation_ids) == len(rates) == len(lengths_of_stay) == len(occupancies), "Array length mismatch"
    
    # 2. VARIABLES
    # Decision variables for rates
    r = {i: model.addVar(vtype=GRB.CONTINUOUS, name=f"rate_{i}", lb=minimum_rate) 
         for i in reservation_ids}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize total revenue
    model.setObjective(gp.quicksum(r[i] * lengths_of_stay[i-1] for i in reservation_ids), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS
    # Minimum rate constraint
    model.addConstrs((r[i] >= minimum_rate for i in reservation_ids), name="min_rate")
    
    # Room capacity constraint
    model.addConstrs((occupancies[i-1] <= room_capacity for i in reservation_ids), name="room_capacity")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in reservation_ids:
            print(f"Rate for reservation {i}: {r[i].x:.2f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    else:
        print("Optimization ended with status", model.status)

# Run the optimization
optimize_inn_revenue()