# Complete Optimization Problem and Solution: journal_committee

## 1. Problem Context and Goals

### Context  
The publishing company is focused on optimizing the allocation of editors to journals to enhance overall sales performance. The decision-making process involves determining whether an editor is assigned to a specific journal, represented by binary decision variables. The primary operational goal is to maximize the total sales of journals by strategically assigning editors, while adhering to workload limitations and ensuring editors are only assigned to journals within their area of expertise. The business configuration includes a critical parameter: the maximum number of journals an editor can manage, which serves as a constraint to prevent overburdening editors. This setup ensures that the optimization problem remains linear, avoiding complex relationships such as variable products or divisions. The data used in this process reflects current operational realities, providing a realistic basis for decision-making.

### Goals  
The optimization goal is to maximize the total sales generated by the journals. This is achieved by assigning editors to journals in a manner that maximizes the sum of sales associated with each journal assignment. The success of this optimization is measured by the total sales figure, which is directly influenced by the assignment decisions. The objective is clearly defined in linear terms, focusing on maximizing the sum of sales contributions from each journal-editor assignment.

## 2. Constraints    

The optimization process is subject to several constraints to ensure feasibility and alignment with business requirements:

- Each editor can be assigned to a limited number of journals, specifically up to the maximum number of journals they can handle. This constraint ensures that editors are not overburdened and can maintain quality in their work.
  
- Editors can only be assigned to journals for which they are qualified based on the journal's theme. This constraint ensures that editors are matched with journals that align with their expertise, maintaining the quality and relevance of editorial work.

These constraints are expressed in a manner that naturally leads to linear mathematical formulations, avoiding any nonlinear relationships.

## 3. Available Data  

### Database Schema  
```sql
-- Iteration 2 Database Schema
-- Objective: Incorporated Sales_journal data into the schema for objective coefficient mapping and updated business configuration logic for scalar parameters.

CREATE TABLE journal_committee (
  Editor_ID INTEGER,
  Journal_ID INTEGER
);

CREATE TABLE editor_qualifications (
  Editor_ID INTEGER,
  Theme STRING
);

CREATE TABLE journal_sales (
  Journal_ID INTEGER,
  Sales FLOAT
);
```

### Data Dictionary  
The data dictionary provides a comprehensive overview of the tables and columns, detailing their business purposes and roles in the optimization process:

- **Journal Committee Table**: This table records the assignments of editors to journals. It plays a crucial role in defining the decision variables for the optimization problem, where each entry indicates a potential assignment.

  - **Editor_ID**: Serves as a unique identifier for each editor, crucial for tracking assignments.
  - **Journal_ID**: Serves as a unique identifier for each journal, essential for mapping assignments.

- **Editor Qualifications Table**: This table captures the themes for which each editor is qualified, ensuring that assignments are made within the editor's area of expertise.

  - **Editor_ID**: Identifies each editor, linking them to their qualifications.
  - **Theme**: Represents the specific theme an editor is qualified to handle, used to enforce qualification constraints.

- **Journal Sales Table**: This table contains sales data for each journal, which is used as coefficients in the objective function to maximize total sales.

  - **Journal_ID**: Identifies each journal, linking it to its sales data.
  - **Sales**: Represents the sales figures associated with each journal, serving as a key component in the optimization objective.

### Current Stored Values  
```sql
-- Iteration 2 Realistic Data
-- Generated by triple expert (business + data + optimization)
-- Values were determined based on typical workload capacities for editors and realistic sales figures for journals in a mid-sized publishing company.

-- Realistic data for journal_committee
INSERT INTO journal_committee (Editor_ID, Journal_ID) VALUES (1, 101);
INSERT INTO journal_committee (Editor_ID, Journal_ID) VALUES (2, 102);
INSERT INTO journal_committee (Editor_ID, Journal_ID) VALUES (3, 103);

-- Realistic data for editor_qualifications
INSERT INTO editor_qualifications (Editor_ID, Theme) VALUES (1, 'Science');
INSERT INTO editor_qualifications (Editor_ID, Theme) VALUES (2, 'Arts');
INSERT INTO editor_qualifications (Editor_ID, Theme) VALUES (3, 'Technology');

-- Realistic data for journal_sales
INSERT INTO journal_sales (Journal_ID, Sales) VALUES (101, 1200.0);
INSERT INTO journal_sales (Journal_ID, Sales) VALUES (102, 1800.0);
INSERT INTO journal_sales (Journal_ID, Sales) VALUES (103, 1600.0);
```

## 4. Mathematical Optimization Formulation

#### Decision Variables
- Let \( x_{ij} \) be a binary decision variable where \( x_{ij} = 1 \) if editor \( i \) is assigned to journal \( j \), and \( x_{ij} = 0 \) otherwise.

#### Objective Function
Maximize the total sales from journal assignments:
\[
\text{Maximize } \sum_{i,j} \text{Sales}_j \times x_{ij}
\]
where \(\text{Sales}_j\) is the sales figure for journal \( j \) from the `journal_sales` table.

#### Constraints
1. **Editor Workload Constraint**: Each editor can be assigned to a limited number of journals. Let \( M_i \) be the maximum number of journals editor \( i \) can handle.
   \[
   \sum_{j} x_{ij} \leq M_i \quad \forall i
   \]

2. **Qualification Constraint**: Editors can only be assigned to journals for which they are qualified. This requires matching the editor's qualifications with the journal's theme.
   \[
   x_{ij} = 0 \quad \text{if editor } i \text{ is not qualified for journal } j
   \]

3. **Binary Constraint**: Each decision variable is binary.
   \[
   x_{ij} \in \{0, 1\} \quad \forall i, j
   \]

Data Source Verification:
- **Sales_j**: From `journal_sales.Sales`
- **Qualification Constraint**: Derived from `editor_qualifications` and the thematic requirements of each journal.
- **Editor Workload Constraint**: Business configuration parameter \( M_i \) (not explicitly provided in the data, assumed to be a known parameter for each editor).

This formulation provides a complete and linear model for optimizing the assignment of editors to journals, ensuring that all constraints are respected and the objective of maximizing total sales is achieved.

## 5. Gurobipy Implementation

```python
# Complete GUROBIPY implementation

import gurobipy as gp
from gurobipy import GRB

def optimize_editor_journal_assignment():
    """Optimize the assignment of editors to journals to maximize sales."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("journal_committee")
    
    # Data: Example data based on the problem context
    editors = [1, 2, 3]
    journals = [101, 102, 103]
    sales = {101: 1200.0, 102: 1800.0, 103: 1600.0}
    qualifications = {
        1: ['Science'],
        2: ['Arts'],
        3: ['Technology']
    }
    journal_themes = {
        101: 'Science',
        102: 'Arts',
        103: 'Technology'
    }
    max_journals_per_editor = {1: 2, 2: 2, 3: 2}
    
    # Validate data lengths
    assert len(editors) == len(max_journals_per_editor), "Editor data length mismatch"
    assert len(journals) == len(sales) == len(journal_themes), "Journal data length mismatch"
    
    # 2. VARIABLES
    # Binary decision variables for editor-journal assignments
    x = model.addVars(editors, journals, vtype=GRB.BINARY, name="x")
    
    # 3. OBJECTIVE FUNCTION
    # Maximize total sales from journal assignments
    model.setObjective(gp.quicksum(sales[j] * x[i, j] for i in editors for j in journals), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS
    
    # Editor Workload Constraint
    for i in editors:
        model.addConstr(gp.quicksum(x[i, j] for j in journals) <= max_journals_per_editor[i], name=f"workload_{i}")
    
    # Qualification Constraint
    for i in editors:
        for j in journals:
            if journal_themes[j] not in qualifications[i]:
                model.addConstr(x[i, j] == 0, name=f"qualification_{i}_{j}")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in editors:
            for j in journals:
                if x[i, j].x > 1e-6:
                    print(f"Editor {i} assigned to Journal {j}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
optimize_editor_journal_assignment()
```

### Execution Results
**Status**: OPTIMAL
**Optimal Value**: 4600.0
**Execution Time**: 1.44 seconds
**Reliability**: high
**Analysis**: Gurobipy found an optimal solution quickly with a consistent optimal value, indicating high reliability.

## 6. DOCplex Implementation

```python
# Complete DOCPLEX implementation

from docplex.mp.model import Model

def optimize_editor_journal_assignment():
    """Optimize the assignment of editors to journals to maximize sales."""
    
    # 1. MODEL & DATA SETUP
    mdl = Model(name="journal_committee")
    
    # Data from the problem context
    editors = [1, 2, 3]
    journals = [101, 102, 103]
    sales = {101: 1200.0, 102: 1800.0, 103: 1600.0}
    qualifications = {
        1: ['Science'],
        2: ['Arts'],
        3: ['Technology']
    }
    journal_themes = {
        101: 'Science',
        102: 'Arts',
        103: 'Technology'
    }
    max_journals_per_editor = {1: 2, 2: 2, 3: 2}  # Example workload limits
    
    # CRITICAL: Validate array lengths to prevent IndexError
    assert len(editors) == len(max_journals_per_editor), "Array length mismatch"
    safe_range_editors = range(len(editors))
    safe_range_journals = range(len(journals))
    
    # 2. VARIABLES
    # Binary decision variables for editor-journal assignments
    x = {(i, j): mdl.binary_var(name=f"x_{i}_{j}") for i in editors for j in journals}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize total sales from journal assignments
    objective = mdl.sum(sales[j] * x[i, j] for i in editors for j in journals)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Editor Workload Constraint
    for i in editors:
        mdl.add_constraint(mdl.sum(x[i, j] for j in journals) <= max_journals_per_editor[i], ctname=f"workload_{i}")
    
    # Qualification Constraint
    for i in editors:
        for j in journals:
            if journal_themes[j] not in qualifications[i]:
                mdl.add_constraint(x[i, j] == 0, ctname=f"qualification_{i}_{j}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in editors:
            for j in journals:
                value = solution.get_value(x[i, j])
                if value > 1e-6:
                    print(f"x[{i},{j}] = {value:.3f}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
    
    return mdl

# Run the optimization
optimize_editor_journal_assignment()
```

### Execution Results
**Status**: OPTIMAL
**Optimal Value**: 4600.0
**Execution Time**: 8.76 seconds
**Reliability**: high
**Analysis**: DOCplex also found an optimal solution with the same optimal value, though it took longer than Gurobipy.

## 7. Pyomo Implementation

```python
# Complete PYOMO implementation

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def journal_committee_optimization():
    """Optimize editor assignments to journals to maximize sales."""
    
    # 1. MODEL CREATION
    model = pyo.ConcreteModel()
    
    # 2. DATA SETUP
    editors = [1, 2, 3]
    journals = [101, 102, 103]
    sales = {101: 1200.0, 102: 1800.0, 103: 1600.0}
    qualifications = {
        1: ['Science'],
        2: ['Arts'],
        3: ['Technology']
    }
    themes = {
        101: 'Science',
        102: 'Arts',
        103: 'Technology'
    }
    max_journals_per_editor = {1: 2, 2: 2, 3: 2}
    
    # CRITICAL: Validate data consistency
    assert len(editors) == len(max_journals_per_editor), "Editor data mismatch"
    assert len(journals) == len(sales), "Journal data mismatch"
    
    # 3. SETS
    model.I = pyo.Set(initialize=editors)
    model.J = pyo.Set(initialize=journals)
    
    # 4. PARAMETERS
    model.sales = pyo.Param(model.J, initialize=sales)
    model.max_journals = pyo.Param(model.I, initialize=max_journals_per_editor)
    
    # 5. VARIABLES
    model.x = pyo.Var(model.I, model.J, within=pyo.Binary)
    
    # 6. OBJECTIVE FUNCTION
    def obj_rule(model):
        return sum(model.sales[j] * model.x[i, j] for i in model.I for j in model.J)
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.maximize)
    
    # 7. CONSTRAINTS
    
    # Editor Workload Constraint
    def workload_rule(model, i):
        return sum(model.x[i, j] for j in model.J) <= model.max_journals[i]
    model.workload_constraint = pyo.Constraint(model.I, rule=workload_rule)
    
    # Qualification Constraint
    def qualification_rule(model, i, j):
        if themes[j] not in qualifications[i]:
            return model.x[i, j] == 0
        return pyo.Constraint.Skip
    model.qualification_constraint = pyo.Constraint(model.I, model.J, rule=qualification_rule)
    
    # 8. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')
    
    # Optional: Set solver options
    solver.options['TimeLimit'] = 300  # 5 minutes
    solver.options['MIPGap'] = 0.01    # 1% gap
    
    # Solve the model
    results = solver.solve(model, tee=True)
    
    # 9. RESULT PROCESSING
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print("Optimal solution found!")
        print(f"Optimal value: {pyo.value(model.objective)}")
        
        # Extract variable values
        print("\nAssignment of editors to journals:")
        for i in model.I:
            for j in model.J:
                if pyo.value(model.x[i, j]) > 0.5:  # Binary variable, check if assigned
                    print(f"Editor {i} assigned to Journal {j}")
        
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")
    
    return model

# Run the optimization
journal_committee_optimization()
```

### Execution Results
**Status**: OPTIMAL
**Optimal Value**: 4600.0
**Execution Time**: 4.87 seconds
**Reliability**: high
**Analysis**: Pyomo achieved the same optimal solution, with execution time between Gurobipy and DOCplex.

## 8. Cross-Solver Analysis and Final Recommendation

### Solver Results Comparison

| Solver | Status | Optimal Value | Execution Time | Decision Variables | Retry Attempt |
|--------|--------|---------------|----------------|-------------------|---------------|
| Gurobipy | OPTIMAL | 4600.00 | 1.44s | N/A | N/A |
| Docplex | OPTIMAL | 4600.00 | 8.76s | N/A | N/A |
| Pyomo | OPTIMAL | 4600.00 | 4.87s | N/A | N/A |

### Solver Consistency Analysis
**Result**: All solvers produced consistent results ✓
**Consistent Solvers**: gurobipy, docplex, pyomo
**Majority Vote Optimal Value**: 4600.0

### Final Recommendation
**Recommended Optimal Value**: 4600.0
**Confidence Level**: HIGH
**Preferred Solver(s)**: gurobipy
**Reasoning**: Gurobipy is preferred due to its faster execution time while maintaining high reliability and consistency with other solvers.

### Business Interpretation
**Overall Strategy**: The optimal assignment of editors to journals maximizes sales, achieving a total sales value of 4600.0.
**Objective Value Meaning**: The optimal objective value of 4600.0 represents the maximum achievable sales from the optimal assignment of editors to journals.
**Resource Allocation Summary**: Editors should be assigned to journals in a way that maximizes sales while respecting workload and qualification constraints.
**Implementation Recommendations**: Implement the solution by assigning editors to the journals as per the optimal decision variables, ensuring that all constraints are adhered to.