# Complete GUROBIPY implementation

import gurobipy as gp
from gurobipy import GRB

def optimize_production():
    """Optimize production quantities to maximize revenue"""

    # 1. MODEL & DATA SETUP
    model = gp.Model("manufactory_1")
    
    # Prices and capacities from the database
    prices = [12.99, 18.49, 22.00]
    manufacturer_capacities = [5000, 7000, 8000]
    total_capacity = sum(manufacturer_capacities)
    
    # CRITICAL: Validate array lengths before loops
    n_products = len(prices)
    assert len(manufacturer_capacities) == n_products, "Array length mismatch"
    
    # 2. VARIABLES
    # Variable dictionaries for production quantities
    x = {i: model.addVar(vtype=GRB.CONTINUOUS, name=f"x_{i+1}", lb=0) 
         for i in range(n_products)}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize total revenue
    model.setObjective(gp.quicksum(prices[i] * x[i] for i in range(n_products)), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Total production capacity constraint
    model.addConstr(gp.quicksum(x[i] for i in range(n_products)) <= total_capacity, name="total_capacity")
    
    # Manufacturer-specific capacity constraints
    for i in range(n_products):
        model.addConstr(x[i] <= manufacturer_capacities[i], name=f"capacity_{i+1}")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in range(n_products):
            if x[i].x > 1e-6:
                print(f"x[{i+1}] = {x[i].x:.3f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
optimize_production()