# Complete GUROBIPY implementation

import gurobipy as gp
from gurobipy import GRB

def optimize_furniture_production():
    """Optimize furniture production to maximize profit"""

    # 1. MODEL & DATA SETUP
    model = gp.Model("furniture_optimization")

    # Data: Profit margins and capacities
    profit_margins = [60, 90, 70]  # Derived from market_rate - price_in_dollar
    total_production_capacity = 40  # Example capacity
    total_sales_capacity = 35  # Example capacity

    # Number of furniture items
    n_items = len(profit_margins)

    # CRITICAL: Validate array lengths before loops
    assert len(profit_margins) == n_items, "Array length mismatch"

    # 2. VARIABLES
    # Decision variables for quantities of each furniture item
    quantities = model.addVars(n_items, vtype=GRB.INTEGER, name="quantity", lb=0)

    # 3. OBJECTIVE FUNCTION
    # Maximize total profit
    model.setObjective(gp.quicksum(profit_margins[i] * quantities[i] for i in range(n_items)), GRB.MAXIMIZE)

    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS

    # Total production capacity constraint
    model.addConstr(gp.quicksum(quantities[i] for i in range(n_items)) <= total_production_capacity, name="production_capacity")

    # Total sales capacity constraint
    model.addConstr(gp.quicksum(quantities[i] for i in range(n_items)) <= total_sales_capacity, name="sales_capacity")

    # 5. SOLVING & RESULTS
    model.optimize()

    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in range(n_items):
            if quantities[i].x > 1e-6:
                print(f"quantity[{i}] = {quantities[i].x:.0f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")

    return model

# Run the optimization
optimize_furniture_production()