# Complete DOCPLEX implementation

from docplex.mp.model import Model

def optimize_medicine_enzyme_interaction():
    """Optimize the selection of medicine-enzyme interactions for research."""
    
    # 1. MODEL & DATA SETUP
    mdl = Model(name="medicine_enzyme_interaction")
    
    # Data from the database
    interaction_ids = [1, 2, 3]
    impact_coefficients = [1.5, 0.9, 1.2]
    interaction_costs = [1500, 800, 1200]
    resource_usage = [25, 10, 18]
    
    # Business constraints
    total_budget = 2500  # Example budget
    total_resources = 40  # Example resource limit
    
    # CRITICAL: Validate array lengths to prevent IndexError
    assert len(interaction_ids) == len(impact_coefficients) == len(interaction_costs) == len(resource_usage), "Array length mismatch"
    safe_range = range(len(interaction_ids))  # Safe indexing
    
    # 2. VARIABLES
    # Binary decision variables for each interaction
    x = {i: mdl.binary_var(name=f"x_{interaction_ids[i]}") for i in safe_range}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the total impact of selected interactions
    objective = mdl.sum(impact_coefficients[i] * x[i] for i in safe_range)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Budget constraint
    total_cost = mdl.sum(interaction_costs[i] * x[i] for i in safe_range)
    mdl.add_constraint(total_cost <= total_budget, ctname="budget_constraint")
    
    # Resource constraint
    total_resource_usage = mdl.sum(resource_usage[i] * x[i] for i in safe_range)
    mdl.add_constraint(total_resource_usage <= total_resources, ctname="resource_constraint")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in safe_range:
            value = solution.get_value(x[i])
            if value > 1e-6:
                print(f"x[{interaction_ids[i]}] = {value:.3f}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
    
    return mdl

# Run the optimization
optimize_medicine_enzyme_interaction()