# Complete GUROBIPY implementation

import gurobipy as gp
from gurobipy import GRB

def optimize_medicine_enzyme_interaction():
    """Optimize medicine-enzyme interactions for research impact"""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("medicine_enzyme_interaction")
    
    # Data from the database
    interaction_ids = [1, 2, 3]
    impact_coefficients = [1.5, 0.9, 1.2]
    interaction_costs = [1500, 800, 1200]
    resource_usage = [25, 10, 18]
    
    # Business constraints
    total_budget = 2500  # Example budget
    total_resources = 40  # Example resources
    
    # CRITICAL: Validate array lengths before loops
    assert len(interaction_ids) == len(impact_coefficients) == len(interaction_costs) == len(resource_usage), "Array length mismatch"
    
    # 2. VARIABLES
    # Binary decision variables for each interaction
    x = {i: model.addVar(vtype=GRB.BINARY, name=f"x_{i}") for i in interaction_ids}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the total impact of selected interactions
    model.setObjective(gp.quicksum(impact_coefficients[i-1] * x[i] for i in interaction_ids), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Budget constraint
    model.addConstr(gp.quicksum(interaction_costs[i-1] * x[i] for i in interaction_ids) <= total_budget, name="budget_constraint")
    
    # Resource constraint
    model.addConstr(gp.quicksum(resource_usage[i-1] * x[i] for i in interaction_ids) <= total_resources, name="resource_constraint")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in interaction_ids:
            if x[i].x > 1e-6:
                print(f"Interaction {i} selected: {x[i].x:.0f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
optimize_medicine_enzyme_interaction()