# Complete DOCPLEX implementation

from docplex.mp.model import Model

def optimize_movie_promotion():
    """Optimize promotional budget allocation for movies to maximize rating improvement."""
    
    # 1. MODEL & DATA SETUP
    mdl = Model(name="movie_promotion")
    
    # Data: Sensitivity values and minimum budget requirements
    sensitivity_values = [0.15, 0.25, 0.1]
    min_budgets = [20000, 30000, 15000]
    total_budget = 65000
    
    # CRITICAL: Validate array lengths to prevent IndexError
    assert len(sensitivity_values) == len(min_budgets), "Array length mismatch"
    safe_range = range(len(sensitivity_values))  # Safe indexing
    
    # 2. VARIABLES
    # Variable dictionaries for budget allocation
    budget_allocation = {i: mdl.continuous_var(name=f"budget_{i}", lb=0) for i in safe_range}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the expected increase in movie ratings
    objective = mdl.sum(sensitivity_values[i] * budget_allocation[i] for i in safe_range)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Total budget constraint
    total_allocation = mdl.sum(budget_allocation[i] for i in safe_range)
    mdl.add_constraint(total_allocation <= total_budget, ctname="total_budget_limit")
    
    # Minimum budget constraints for each movie
    for i in safe_range:
        mdl.add_constraint(budget_allocation[i] >= min_budgets[i], ctname=f"min_budget_{i}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in safe_range:
            value = solution.get_value(budget_allocation[i])
            print(f"Budget allocation for movie {i+1}: {value:.2f}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
    
    return mdl

# Run the optimization
optimize_movie_promotion()