# Complete PYOMO implementation

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def museum_visit_optimization():
    """Optimize staff allocation across museums to maximize visitor satisfaction."""
    
    # 1. MODEL CREATION
    model = pyo.ConcreteModel()
    
    # 2. DATA SETUP
    # Assuming total available staff is provided as a parameter
    total_available_staff = 30
    
    # Coefficients and bounds from the database
    satisfaction_coefficients = [1.3, 1.6, 1.2]
    cost_coefficients = [0.9, 1.1, 0.8]
    minimum_staff_required = [5, 7, 4]
    maximum_staff_capacity = [15, 20, 10]
    
    # Validate array lengths
    assert len(satisfaction_coefficients) == len(cost_coefficients) == len(minimum_staff_required) == len(maximum_staff_capacity) == 3, "Array length mismatch"
    
    # 3. SETS
    model.M = pyo.RangeSet(1, 3)  # 3 museums
    
    # 4. PARAMETERS
    model.satisfaction_coefficient = pyo.Param(model.M, initialize={i+1: satisfaction_coefficients[i] for i in range(3)})
    model.cost_coefficient = pyo.Param(model.M, initialize={i+1: cost_coefficients[i] for i in range(3)})
    model.minimum_staff = pyo.Param(model.M, initialize={i+1: minimum_staff_required[i] for i in range(3)})
    model.maximum_staff = pyo.Param(model.M, initialize={i+1: maximum_staff_capacity[i] for i in range(3)})
    
    # 5. VARIABLES
    model.x = pyo.Var(model.M, within=pyo.NonNegativeIntegers)
    
    # 6. OBJECTIVE FUNCTION
    def obj_rule(model):
        return sum((model.satisfaction_coefficient[i] - model.cost_coefficient[i]) * model.x[i] for i in model.M)
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.maximize)
    
    # 7. CONSTRAINTS
    # Total staff constraint
    def total_staff_rule(model):
        return sum(model.x[i] for i in model.M) <= total_available_staff
    model.total_staff_constraint = pyo.Constraint(rule=total_staff_rule)
    
    # Minimum and maximum staff constraints
    def staff_bounds_rule(model, i):
        return (model.minimum_staff[i], model.x[i], model.maximum_staff[i])
    model.staff_bounds_constraint = pyo.Constraint(model.M, rule=staff_bounds_rule)
    
    # 8. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')
    
    # Optional: Set solver options
    solver.options['TimeLimit'] = 300  # 5 minutes
    solver.options['MIPGap'] = 0.01    # 1% gap
    
    # Solve the model
    results = solver.solve(model, tee=True)  # tee=True shows solver output
    
    # 9. RESULT PROCESSING
    # Check solver status
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print("Optimal solution found!")
        print(f"Optimal value: {pyo.value(model.objective):.3f}")
        
        # Extract variable values
        print("\nVariable values:")
        for i in model.M:
            x_val = pyo.value(model.x[i])
            if x_val > 1e-6:  # Only print non-zero values
                print(f"x[{i}] = {x_val}")
        
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")
    
    return model

# Run the optimization
museum_visit_optimization()