# Complete DOCPLEX implementation

from docplex.mp.model import Model

def optimize_music_storage():
    """Optimize music storage to maximize total rating under constraints."""
    
    # 1. MODEL & DATA SETUP
    mdl = Model(name="music_optimization")
    
    # Data from the problem context
    ratings = [4.5, 3.8, 5.0]
    file_sizes = [5000, 3000, 7000]
    resolutions = [1080, 720, 480]
    
    # Assumed constraints
    total_storage_capacity = 10000
    minimum_resolution_quality = 1500
    
    # CRITICAL: Validate array lengths to prevent IndexError
    assert len(ratings) == len(file_sizes) == len(resolutions), "Array length mismatch"
    safe_range = range(min(len(ratings), len(file_sizes), len(resolutions)))  # Safe indexing
    
    # 2. VARIABLES
    x = {i: mdl.binary_var(name=f"x_{i}") for i in safe_range}
    
    # 3. OBJECTIVE FUNCTION
    objective = mdl.sum(ratings[i] * x[i] for i in safe_range)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Storage Capacity Constraint
    storage_constraint = mdl.sum(file_sizes[i] * x[i] for i in safe_range)
    mdl.add_constraint(storage_constraint <= total_storage_capacity, ctname="storage_capacity")
    
    # Resolution Quality Constraint
    resolution_constraint = mdl.sum(resolutions[i] * x[i] for i in safe_range)
    mdl.add_constraint(resolution_constraint >= minimum_resolution_quality, ctname="resolution_quality")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in safe_range:
            value = solution.get_value(x[i])
            if value > 1e-6:
                print(f"x[{i}] = {value:.3f}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
    
    return mdl

# Run the optimization
optimize_music_storage()