# Complete GUROBIPY implementation - Retry Attempt 1

import gurobipy as gp
from gurobipy import GRB

def music_festival_optimization():
    """Optimize artist selection for maximum exposure at a music festival."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("music_festival")
    
    # Data from the problem statement
    artist_ids = [1, 2, 3]
    weeks_on_top = [12, 8, 15]
    availability_status = [True, False, True]
    capacity = 2
    
    # Validate array lengths
    assert len(artist_ids) == len(weeks_on_top) == len(availability_status), "Array length mismatch"
    
    # 2. VARIABLES
    # Binary decision variables for each artist
    x = {i: model.addVar(vtype=GRB.BINARY, name=f"x_{artist_ids[i]}") for i in range(len(artist_ids))}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize total weeks on top for selected artists
    model.setObjective(gp.quicksum(weeks_on_top[i] * x[i] for i in range(len(artist_ids))), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS
    # Capacity constraint
    model.addConstr(gp.quicksum(x[i] for i in range(len(artist_ids))) <= capacity, name="capacity_constraint")
    
    # Availability constraints
    for i in range(len(artist_ids)):
        model.addConstr(x[i] <= int(availability_status[i]), name=f"availability_{artist_ids[i]}")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in range(len(artist_ids)):
            if x[i].x > 1e-6:
                print(f"Artist {artist_ids[i]} selected: x[{i}] = {x[i].x:.3f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    else:
        print("Optimization ended with status", model.status)

# Run the optimization
music_festival_optimization()