# Complete DOCPLEX implementation

from docplex.mp.model import Model

def optimize_social_network_campaign():
    """Optimize the targeting strategy for a social network marketing campaign."""
    
    # 1. MODEL & DATA SETUP
    mdl = Model(name="network_2")
    
    # Data from the problem statement
    influence_values = [1.5, 0.9, 1.2]
    targeting_costs = [300.0, 150.0, 250.0]
    budget = 500.0
    
    # Validate array lengths to prevent IndexError
    assert len(influence_values) == len(targeting_costs), "Array length mismatch"
    safe_range = range(len(influence_values))  # Safe indexing
    
    # 2. VARIABLES
    # Binary decision variables for targeting individuals
    x = {i: mdl.binary_var(name=f"x_{i}") for i in safe_range}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the total influence spread across the network
    objective = mdl.sum(influence_values[i] * x[i] for i in safe_range)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS
    # Budget constraint: Total cost of targeting must not exceed the budget
    total_cost = mdl.sum(targeting_costs[i] * x[i] for i in safe_range)
    mdl.add_constraint(total_cost <= budget, ctname="budget_constraint")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in safe_range:
            value = solution.get_value(x[i])
            if value > 1e-6:
                print(f"x[{i}] = {value:.0f}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
    
    return mdl

# Run the optimization
optimize_social_network_campaign()