# Complete DOCPLEX implementation

from docplex.mp.model import Model

def optimize_orchestra_performances():
    """Optimize the number of performances for each show to maximize attendance."""
    
    # 1. MODEL & DATA SETUP
    mdl = Model(name="orchestra_optimization")
    
    # Data from the problem statement
    show_ids = [101, 102, 103]
    attendances = [150, 250, 350]
    max_performances = [5, 10, 15]
    capacities = [200, 300, 400]
    total_conductor_availability = 20
    
    # CRITICAL: Validate array lengths to prevent IndexError
    assert len(show_ids) == len(attendances) == len(max_performances) == len(capacities), "Array length mismatch"
    safe_range = range(len(show_ids))  # Safe indexing
    
    # 2. VARIABLES
    performances = {i: mdl.continuous_var(name=f"P_{show_ids[i]}", lb=0) for i in safe_range}
    
    # 3. OBJECTIVE FUNCTION
    objective = mdl.sum(attendances[i] * performances[i] for i in safe_range)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Total availability of conductors
    total_performances = mdl.sum(performances[i] for i in safe_range)
    mdl.add_constraint(total_performances <= total_conductor_availability, ctname="total_conductor_availability")
    
    # Orchestra performance limits
    for i in safe_range:
        mdl.add_constraint(performances[i] <= max_performances[i], ctname=f"max_performances_{show_ids[i]}")
    
    # Venue capacity constraints
    for i in safe_range:
        mdl.add_constraint(attendances[i] * performances[i] <= capacities[i], ctname=f"venue_capacity_{show_ids[i]}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in safe_range:
            value = solution.get_value(performances[i])
            if value > 1e-6:
                print(f"Performances for Show {show_ids[i]}: {value:.3f}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
    
    return mdl

# Run the optimization
optimize_orchestra_performances()